
/*
 *  nwfsinfo.c
 *
 *  Print the info strings of a server, maybe sometime more.
 *
 *  Copyright (C) 1996 by Volker Lendecke
 *
 */

#include <stdio.h>
#include <unistd.h>
#include "ncplib.h"

static char *progname;

static void usage(void)
{
	fprintf(stderr, "usage: %s [pattern]\n", progname);
}

static void help(void)
{
	printf("\n");
	printf("usage: %s [options]\n", progname);
	printf("\n"
	       "-h             Print this help text\n"
	       "-S server      Server name to be used\n"
	       "\n"
	       "-d             Print Description Strings\n"
	       "-t             Print File Server's time\n"
	       "-i             Print File Server Information\n"
	       "\n");
}

static void print_info(struct ncp_file_server_info *info)
{
	printf("\n");
	printf("Fileservername    %-48.48s\n", info->ServerName);
	printf("\n");
	printf("Version           %d.%d Revision %c\n",
	       info->FileServiceVersion, info->FileServiceSubVersion,
	       info->Revision + 'A');
	printf("Max. Connections  %d\n",
	       info->MaximumServiceConnections);
	printf("currently in use  %d\n",
	       info->ConnectionsInUse);
	printf("peak connections  %d\n",
	       info->MaxConnectionsEverUsed);
	printf("Max. Volumes      %d\n",
	       info->NumberMountedVolumes);
	printf("SFTLevel          %d\n",
	       info->SFTLevel);
	printf("TTSLevel          %d\n",
	       info->TTSLevel);
	printf("Accountversion    %d\n",
	       info->AccountVersion);
	printf("Queueversion      %d\n",
	       info->QueueVersion);
	printf("Printversion      %d\n",
	       info->PrintVersion);
	printf("Virt.Consolvers.  %d\n",
	       info->VirtualConsoleVersion);
	printf("RestrictionLevel  %d\n",
	       info->RestrictionLevel);
	printf("\n");
	return;
}

int main(int argc, char **argv)
{
	struct ncp_conn *conn;
	int opt;
	long err;

	progname = argv[0];

	if ((conn = ncp_initialize(&argc, argv, 0, &err)) == NULL) {
		com_err(argv[0], err, "when initializing");
		return 1;
	}
	while ((opt = getopt(argc, argv, "h?dti")) != EOF) {
		switch (opt) {
		case 'h':
		case '?':
			help();
			break;
		case 'd':
			{
				char strings[512];
				char *s;

				if (ncp_get_file_server_description_strings(conn,
								 strings)
				    != 0) {
					perror("could not get strings");
					ncp_close(conn);
					return 1;
				}
				s = strings;
				while (s < strings + 512) {
					if (strlen(s) == 0) {
						break;
					}
					puts(s);
					s += strlen(s) + 1;
				}
				break;
			}
		case 't':
			{
				time_t t;

				if (ncp_get_file_server_time(conn, &t) != 0) {
					perror("could not get server time");
					ncp_close(conn);
					return 1;
				}
				fputs(ctime(&t), stdout);
				break;
			}
		case 'i':
			{
				struct ncp_file_server_info info;
				if (ncp_get_file_server_information(conn, &info) != 0) {
					perror("Could not get server information");
					ncp_close(conn);
					return 1;
				}
				print_info(&info);
				break;
			}

		default:
			usage();
			goto finished;
		}
	}

      finished:
	ncp_close(conn);
	return 0;
}
