#!/bin/sh
#
# Install script for Digital extensions to HP OpenView
# (c) Copyright 1998, Digital Equipment Corporation
#
##################################################################

##################################################################
# global variables
##################################################################
startdir=`pwd`
ErrLog="/tmp/SWInstallErr.log"
InstallLog="/tmp/SWInstall.log"

##################################################################
# Error handlers
##################################################################
fatal()
{
	echo; echo
	echo "ERROR: $@."
	echo "This was a fatal error.  Installation aborted."
	echo >> $ErrLog
	echo "ERROR: $@." >> $ErrLog
	echo "This was a fatal error.  Installation aborted." >> $ErrLog
	cleanup
	exit 1
}

##################################################################
# User interaction routines
##################################################################
yesno()
{
	while true; do
	$echon "$1 (Y/N)? "
	read ans
	case $ans in
		y|Y)	return 0;;
		n|N)	return 1;;
	esac
	done
}

readln()
{
	$echon 'Press return to continue... '
	read jnk
}

##################################################################
# The routines that follow are all used to make sure that we
# have the correct environment for installation.  Are we root,
# are the tools we need installed, has the OV environment 
# been set.
##################################################################

check_for_root()
{
	echo "Checking that we are super-user" >> $InstallLog
	if [ "`whoami`" != "root" ] ; then
		fatal "Installation must be performed as the super-user!"
	fi
}

check_environment()
{
	echo "Checking that the OpenView environment is set" >> $InstallLog
	test -z "$OV_BIN" && fatal "OpenView environment not set.  Execute ov.envvars.sh and try again"
	test -z "$OV_SNMP_MIBS" && fatal "OpenView environment not set.  Execute ov.envvars.sh and try again"
	test -z "$OV_CONF" && fatal "OpenView environment not set.  Execute ov.envvars.sh and try again"
}

check_for_file()
{
	echo "Checking for file $1" >> $InstallLog
	test -r "$1" && return 0
	return 1
}

check_for_files()
{
    for this_bin
    do
        check_for_file $this_bin ||
          fatal "file '$this_bin' not found"
    done
}

check_for_binary()
{
echo "Checking for binary $1" >> $InstallLog
    testbin=$1
    case "$testbin" in
      /*) test -x "$testbin" && test -f "$testbin"; return;;
      *)
         OLDIFS=$IFS; IFS=:; eval set $PATH; IFS=$OLDIFS
         for this_dir
         do
             test -x "$this_dir/$testbin" &&
             test -f "$this_dir/$testbin" && return 0
         done
         return 1;;
    esac
}

require_binaries()
{
    for this_bin
    do
        check_for_binary $this_bin ||
          fatal "program '$this_bin' not found in PATH"
    done
}

locate_binaries()
{
	require_binaries touch sed awk ln rm ls tr mkdir cat pwd
}

check_for_directory()
{
echo "Checking for directory $1" >> $InstallLog
	test -r "$1" && return 0
	return 1
}

make_directory()
{
	check_for_directory $1 && return
	echo "Making directory $1" >> $InstallLog
	mkdir $1 && return
	fatal "Unable to make directory $1"
}

##################################################################
# Here are a set of test functions for echo.  The purpose is to
# find which one will do the echo without appending a line feed
# They are used by find_echo below for this purpose.
##################################################################
echo_a() { echo      -n "$@";   }
echo_b() { echo         "$@\c"; }
echo_c() { echo      -e "$@\c"; }
echo_d() { /bin/echo -n "$@";   }
echo_e() { /bin/echo    "$@\c"; }
echo_f() { /bin/echo -e "$@\c"; }

##################################################################
# Find out which echo will not append a line feed to the end of
# our string
##################################################################
find_echo()
{
    for i in a b c d e f; do
        test "`echo_$i c``echo_$i a`"  = ca && echon=echo_$i && return
    done
    echon=echo
}

prepare_logs()
{
	if test ! -w $InstallLog; then
		touch "$InstallLog"
	else
		mv $InstallLog $InstallLog.old
		touch "$InstallLog"
	fi
	test -w "$InstallLog" || fatal "Cannot write to file $InstallLog"
	if test ! -w $ErrLog; then
		touch "$ErrLog"
	else
		mv $ErrLog $ErrLog.old
		touch "$ErrLog"
	fi
	test -w "$ErrLog" || fatal "Cannot write to file $ErrLog"
	ts=`date`
	echo "***************************************" >> $InstallLog
	echo "	TIMESTAMP $ts" >> $InstallLog
	echo "***************************************" >> $InstallLog
	echo "***************************************" >> $ErrLog
	echo "	TIMESTAMP $ts" >> $ErrLog
	echo "***************************************" >> $ErrLog
}

##################################################################
# init - performs initialzation for the installation process
# 
##################################################################
init()
{
	
	find_echo
	prepare_logs

	echo "Beginning installation of Digital's HP OpenView intergration" >> $InstallLog
	echo >> $InstallLog
	echo "========== Start INIT Section ==========" >> $InstallLog

	check_for_root
	cd $startdir
	$echon "Initializing..."
	locate_binaries
	check_environment
	echo "Done"

	echo "========== Finish INIT Section ==========" >> $InstallLog
	echo >> $InstallLog
}

##################################################################
# install_mibs
##################################################################
xnmloadmib_status()
{
	case $1 in
		1)	fatal "Invalid command line usage";;
		2)  fatal "Syntax error in the MIB file $2";;
		4)	fatal "MIB name specified by -unload was not loaded";;
		5)  fatal "MIB file $2 could not be accessed";;
		6)  fatal "MIB file $2 specified with -load is not a file";;
		7)  fatal "Error accessing MIB Database";;
		8)  fatal "Error creating MIB Database";;
		9)  fatal "Error creating compiled form MIB Database";;
		*)  fatal "Unknown error";;
	esac
}

install_mibs()
{
	mib_files="snmp_mibs/Standard/RFC1514-HOSTRESOURCES snmp_mibs/Vendor/Digital/svrsys.mib snmp_mibs/Vendor/Digital/svrmgt.mib snmp_mibs/Vendor/Digital/ntcmgt.mib snmp_mibs/Vendor/Digital/svrntc.mib"

	echo "========== Start Install MIB ==========" >> $InstallLog
	echo >> $InstallLog
	check_for_files $mib_files
	echo; $echon "Installing MIBs."
	for mf in $mib_files; 
	do
		$OV_BIN/xnmloadmib -load $mf 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			if test $stat != 3; then
				xnmloadmib_status $stat $mf
			else
				echo "MIB $mf already loaded - replacing" >> $InstallLog
				$OV_BIN/xnmloadmib -replace -load $mf 2>> $ErrLog
				stat=$?
				if test $stat != 0; then
					xnmloadmib_status $stat $mf
				fi
			fi
		fi 
		$echon "."
	done
	$OV_BIN/xnmloadmib -event
	echo "Done" 
	echo "========== Finish Install MIB ==========" >> $InstallLog
	echo >> $InstallLog
}

##################################################################
# install_nt_mib_definitions
##################################################################
edit_oid_to_type()
{
	echo "Editing oid_to_type file" >> $InstallLog
	grep -q "1.3.6.1.4.1.311.1.1.3.1.\*::Windows NT:" oid_to_type
	stat=$?
	if test $stat != 0; then
		sed -e's/1.3.6.1.4.1.311.1.1.3.1::Windows NT:/1.3.6.1.4.1.311.1.1.3.1.*::Windows NT:/' oid_to_type > oid_to_type.tmp
		mv oid_to_type.tmp oid_to_type
	fi
}

edit_oid_to_sym()
{
	echo "Editing oid_to_sym file" >> $InstallLog
	grep -q "1.3.6.1.4.1.311.1.1.3.1.1:Computer:WindowsNT" oid_to_sym
	stat=$?
	if test $stat != 0; then
		sed -e'/1.3.6.1.4.1.311.1.1.3.1:Computer:WindowsNT/a\
1.3.6.1.4.1.311.1.1.3.1.1:Computer:WindowsNT # Windows NT Workstation' oid_to_sym > oid_to_sym.tmp
		mv oid_to_sym.tmp oid_to_sym
	fi

	grep -q "1.3.6.1.4.1.311.1.1.3.1.2:Computer:WindowsNT" oid_to_sym
	stat=$?
	if test $stat != 0; then
		sed -e'/1.3.6.1.4.1.311.1.1.3.1.1:Computer:WindowsNT/a\
1.3.6.1.4.1.311.1.1.3.1.2:Computer:WindowsNT # Windows NT Server' oid_to_sym > oid_to_sym.tmp
		mv oid_to_sym.tmp oid_to_sym
	fi

	grep -q "1.3.6.1.4.1.311.1.1.3.1.3:Computer:WindowsNT" oid_to_sym
	stat=$?
	if test $stat != 0; then
		sed -e'/1.3.6.1.4.1.311.1.1.3.1.2:Computer:WindowsNT/a\
1.3.6.1.4.1.311.1.1.3.1.3:Computer:WindowsNT # Windows NT Domain Controller' oid_to_sym > oid_to_sym.tmp
		mv oid_to_sym.tmp oid_to_sym
	fi
}

install_nt_mib_definitions()
{
	echo "========== Start Install NT MIB Definitions ==========" >> $InstallLog
	echo >> $InstallLog

	cd $OV_CONF
	check_for_files oid_to_type oid_to_sym
	echo "Copying oid_to_type to oid_to_type.orig" >> $InstallLog
	echo "Copying oid_to_sym to oid_to_sym.orig" >> $InstallLog
	cp oid_to_type oid_to_type.orig 2>> $ErrLog
	cp oid_to_sym oid_to_sym.orig 2>> $ErrLog

	edit_oid_to_type
	edit_oid_to_sym

	echo "========== Finish Install NT MIB Definitions ==========" >> $InstallLog
	echo >> $InstallLog
	cd $startdir
}
##################################################################
# install_traps
##################################################################
xnmevents_status()
{
	case $1 in
		*)  fatal "Error loading trap file $2";;
	esac
}

install_traps()
{
	trap_files="conf/C/SWMgrTraps"

	echo "========== Start Install Traps ==========" >> $InstallLog
	echo >> $InstallLog
	check_for_files $trap_files
#
# Create a small shell script to add the full path to the DECTrap binary
#
	echo "#!/bin/sh" > /tmp/DECInst.sh
	echo "sed -e's!DECTrap.exe!$OV_BIN/DEC/DECTrap.exe!' `pwd`/$trap_files > $OV_CONF/SWMgrTraps" >> /tmp/DECInst.sh
	chmod +x /tmp/DECInst.sh
	/tmp/DECInst.sh
	rm /tmp/DECInst.sh

	trap_files="$OV_CONF/SWMgrTraps"

	echo "Installing trap definition files." >> $InstallLog
	echo; $echon "Installing trap definition files."
	for tf in $trap_files; 
	do
		$OV_BIN/xnmevents -load $tf 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			xnmevents_status $stat $tf
		fi
	$echon "."
	done
	$OV_BIN/xnmevents -event 2>> $ErrLog
	echo "Done"

	destdir=$OV_BIN/DEC
	make_directory $destdir
	check_for_directory bin/DEC/HP
	cd bin/DEC/HP

	echo "Copying DECTrap.exe to $destdir" >> $InstallLog
	echo "Copying DECTrap.exe to $destdir"
	cp DECTrap.exe $destdir 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		fatal "Unable to copy DECTrap.exe to $destdir"
	fi
    chmod a+x $destdir/DECTrap.exe 2>> $ErrLog
    stat=$?
    if test $stat != 0; then
        fatal "Unable to chmod a+x $destdir/DECTrap.exe"
    fi

	echo "Installing trap stub files." >> $InstallLog
	echo; $echon "Installing trap stub files."
	for tf in `ls *.sh`;
	do
		cp $tf $OV_BIN/DEC 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			fatal "Unable to copy $tf to $OV_BIN/DEC"
		fi
		chmod a+x $OV_BIN/DEC/$tf 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			echo "Unable to chmod a+x $OV_BIN/DEC/$tf" >> $ErrLog
		fi
		$echon "."
	done
	echo "Done"

	echo "========== Finish Install Traps ==========" >> $InstallLog
	echo >> $InstallLog
	cd $startdir
}
##################################################################
# install_data_coll_oids
##################################################################
install_data_coll_oids()
{
	echo "========== Start Data Collection OIDS ==========" >> $InstallLog
	echo >> $InstallLog

	if test ! -w $OV_CONF/snmpCol.conf; then
		fatal "Unable to find writeable $OV_CONF/snmpcol.conf"
	fi
	echo;
	$echon "Adding OIDS to the data collector."
	echo "MIB .1.3.6.1.2.1.25.3.3.1.2 hrProcessorLoad units INTEGER S" >> $OV_CONF/snmpCol.conf
	$echon "."
	echo "MIB .1.3.6.1.2.1.25.2.3.1.6 hrStorageUsed units INTEGER S" >> $OV_CONF/snmpCol.conf
	$echon "."
	echo "MIB .1.3.6.1.4.1.36.2.18.22.1.8.3.4.1.4 svrPowerSensorReading units INTEGER S" >> $OV_CONF/snmpCol.conf
	$echon "."
	echo "MIB .1.3.6.1.4.1.36.2.18.22.1.8.1.2.1.3 svrThSensorReading units INTEGER S" >> $OV_CONF/snmpCol.conf
	echo "Done"
	echo "========== Finish Data Collection OIDS ==========" >> $InstallLog
	echo >> $InstallLog

}

##################################################################
# install_mib_apps
##################################################################
install_mib_apps()
{
	echo "========== Start Install MIB Applications ==========" >> $InstallLog
	echo >> $InstallLog

	cd $startdir

	mibdir="registration/C/ovmib"
	echo >> $InstallLog
	if test -r $mibdir; then
		cd $mibdir
	else
		fatal "Unable to find $mibdir"
	fi
	
	echo; $echon "Installing MIB Application files."
	for maf in `ls`;
	do
	echo "Copying $maf to $OV_REGISTRATION/C/ovmib" >> $InstallLog
		cp $maf $OV_REGISTRATION/C/ovmib 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			fatal "Unable to copy $maf to $OV_REGISTRATION/C/ovmib"
		fi
	$echon "."
	done
	echo "Done"

	cd $startdir

	mibdir="help/C/ovmib/OVW/Functions"
	echo >> $InstallLog
	if test -r $mibdir; then
		cd $mibdir
	else
		fatal "Unable to find $mibdir"
	fi
	
	echo; $echon "Installing MIB Application Help files."
	for maf in `ls`;
	do
	echo "Copying $maf to $OV_HELP/C/ovmib/OVW/Functions" >> $InstallLog
		cp $maf $OV_HELP/C/ovmib/OVW/Functions 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			fatal "Unable to copy $maf to $OV_HELP/C/ovmib/OVW/Functions"
		fi
	$echon "."
	done
	echo "Done"
	cd $startdir

	echo "========== Finish Install MIB Applications ==========" >> $InstallLog
	echo >> $InstallLog
}

##################################################################
# install_applications
##################################################################
install_applications()
{
	echo "========== Start Install Application files  ==========" >> $InstallLog
	echo >> $InstallLog
	echo

	destdir=$OV_BIN/DEC
	make_directory $destdir

	echo "Copying DECDiscover.exe to $destdir" >> $InstallLog
	echo "Copying DECDiscover.exe to $destdir"
	cp bin/DEC/HP/DECDiscover.exe $destdir 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		fatal "Unable to copy DECDiscover.exe to $destdir"
	fi
	chmod a+x $destdir/DECDiscover.exe 2>> $ErrLog
    stat=$?
    if test $stat != 0; then
        fatal "Unable to chmod a+x $destdir/DECDiscover.exe"
    fi

	cd $startdir

	destdir=$OV_REGISTRATION/C/DEC
	make_directory $destdir
	check_for_directory registration/C/DEC
	cd registration/C/DEC

	echo "Copying DECDiscover.reg to $destdir" >> $InstallLog
	echo "Copying DECDiscover.reg to $destdir"
	cp DECDiscover.reg $destdir 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		fatal "Unable to copy DECDiscover.reg to $destdir"
	fi
#
# Create a small shell script to add the full path to the DECTrap binary
#
	echo "#!/bin/sh" > /tmp/DECInst.sh
	echo "sed -e's!DECDiscover.exe!$OV_BIN/DEC/DECDiscover.exe!' `pwd`/DECDiscover.reg > $OV_REGISTRATION/C/DEC/DECDiscover.reg" >> /tmp/DECInst.sh
	chmod +x /tmp/DECInst.sh
	/tmp/DECInst.sh
	rm /tmp/DECInst.sh
	cd $startdir

	destdir=$OV_FIELDS/C/DEC
	make_directory $destdir
	check_for_directory fields/C/DEC
	cd fields/C/DEC

	echo "Copying DECDiscoverFields.reg to $destdir" >> $InstallLog
	echo "Copying DECDiscoverFields.reg to $destdir"
	cp DECDiscoverFields.reg $destdir 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		fatal "Unable to copy DECDiscoverFields.reg to $destdir"
	fi

	$OV_BIN/ovw -fields >> $InstallLog

	echo "========== Finish Install Application files  ==========" >> $InstallLog
	echo >> $InstallLog
	cd $startdir
}

##################################################################
# install_symbols
##################################################################
install_symbols()
{
	echo "========== Start Install Symbol Files  ==========" >> $InstallLog
	echo >> $InstallLog
	echo
	
	destdir=$OV_SYMBOLS/C/DEC
	make_directory $destdir

	echo "Copying symbol registration file DECSymbols.reg to $destdir" >> $InstallLog
	echo "Copying symbol registration file DECSymbols.reg to $destdir"
	cp symbols/C/DEC/DECSymbols.reg $destdir 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		fatal "Unable to copy symbol registration file DECSymbols.reg to $destdir"
	fi

	destdir=$OV_BITMAPS/C/DEC
	make_directory $destdir

	cd bitmaps/C/DEC
	echo
	echo "Copying symbol files to $destdir" >> $InstallLog
	$echon "Copying symbol files to $destdir."
	for sf in `ls *`;
	do
		cp $sf $destdir 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			fatal "Unable to copy symbol file $sf to $destdir"
		fi
		$echon "."
	done
	echo "Done"

	echo "========== Finish Install Symbol Files  ==========" >> $InstallLog
	echo >> $InstallLog
	cd $startdir
}

##################################################################
# install_uninstall
##################################################################
install_uninstall()
{
	echo "========== Start Install Uninstall  ==========" >> $InstallLog
	echo >> $InstallLog
	echo
	
	cd $startdir
	destdir=$OV_BIN/DEC
	make_directory $destdir

	echo "Copying uninstall file uninstall..sh to $destdir" >> $InstallLog
	echo "Copying uninstall file uninstall..sh to $destdir"
	cp uninstall.sh $destdir 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		fatal "Unable to copy uninstall.sh to $destdir"
	fi
	
	chmod a+x $destdir/uninstall.sh 2>> $ErrLog
   	stat=$?
    if test $stat != 0; then
    	fatal "Unable to chmod a+x $destdir/uninstall.sh"
   	fi
	echo "========== Finish Install Uninstall ==========" >> $InstallLog
	echo >> $InstallLog
	cd $startdir
}

##################################################################
# Welcome
##################################################################
welcome()
{
	echo 
	echo
	echo
	echo
	echo
	echo
	echo "		Digital ServerWORKS Integration for OpenView"
	echo "		(c) Copyright, 1998 Digital Equipment Corporation"
	echo
	echo
	echo
	echo
	echo
	readln
}

##################################################################
# cleanup
##################################################################
cleanup()
{
	echo "========== Installation Ends ==========" >> $InstallLog
	echo >> $InstallLog
	echo; echo;
	if yesno 'Do you want to view the installtion log'; then
		cat $InstallLog | more
		echo
		readln
	fi
	if (test -f "$InstallLog") then
		if yesno 'Do you want to delete the install log'; then
			rm "$InstallLog"
		fi
	fi
	echo
	echo
	if yesno 'Do you want to view the error log'; then
		cat $ErrLog | more
		echo
		readln
	fi
	if (test -f "$ErrLog") then
		if yesno 'Do you want to delete the error log'; then
			rm "$ErrLog"
		fi
	fi
}

##################################################################
# main()
##################################################################

clear
init
clear
welcome
clear
echo
install_mibs
install_nt_mib_definitions
install_traps
install_data_coll_oids
install_mib_apps
install_applications
install_symbols
install_uninstall
cleanup

