#!/bin/sh
#
# UnInstall script for Digital extensions to HP OpenView
# (c) Copyright 1998, Digital Equipment Corporation
#
##################################################################

##################################################################
# global variables
##################################################################
startdir=`pwd`
ErrLog="/tmp/SWUNInstallErr.log"
InstallLog="/tmp/SWUNInstall.log"

##################################################################
# Error handlers
##################################################################
fatal()
{
	echo; echo
	echo "ERROR: $@."
	echo "This was a fatal error.  UnInstallation aborted."
	echo >> $ErrLog
	echo "ERROR: $@." >> $ErrLog
	echo "This was a fatal error.  UnInstallation aborted." >> $ErrLog
	cleanup
	exit 1
}

##################################################################
# User interaction routines
##################################################################
yesno()
{
	while true; do
	$echon "$1 (Y/N)? "
	read ans
	case $ans in
		y|Y)	return 0;;
		n|N)	return 1;;
	esac
	done
}

readln()
{
	$echon 'Press return to continue... '
	read jnk
}

##################################################################
# The routines that follow are all used to make sure that we
# have the correct environment for uninstallation.  Are we root,
# are the tools we need installed, has the OV environment 
# been set.
##################################################################

check_for_root()
{
echo "Checking that we are super-user" >> $InstallLog
	if [ "`whoami`" != "root" ] ; then
		fatal "UnInstallation must be performed as the super-user!"
	fi
}

check_environment()
{
echo "Checking that the OpenView environment is set" >> $InstallLog
	test -z "$OV_BIN" && fatal "OpenView environment not set.  Execute ov.envvars.sh and try again"
	test -z "$OV_SNMP_MIBS" && fatal "OpenView environment not set.  Execute ov.envvars.sh and try again"
	test -z "$OV_CONF" && fatal "OpenView environment not set.  Execute ov.envvars.sh and try again"
}

check_for_file()
{
echo "Checking for file $1" >> $InstallLog
	test -r "$1" && return
	return 1
}

check_for_files()
{
    for this_bin
    do
        check_for_file $this_bin ||
          fatal "file '$this_bin' not found"
    done
}

check_for_binary()
{
echo "Checking for binary $1" >> $InstallLog
    testbin=$1
    case "$testbin" in
      /*) test -x "$testbin" && test -f "$testbin"; return;;
      *)
         OLDIFS=$IFS; IFS=:; eval set $PATH; IFS=$OLDIFS
         for this_dir
         do
             test -x "$this_dir/$testbin" &&
             test -f "$this_dir/$testbin" && return 0
         done
         return 1;;
    esac
}

require_binaries()
{
    for this_bin
    do
        check_for_binary $this_bin ||
          fatal "program '$this_bin' not found in PATH"
    done
}

locate_binaries()
{
	require_binaries touch sed echo read rm ls cat pwd
}

##################################################################
# Here are a set of test functions for echo.  The purpose is to
# find which one will do the echo without appending a line feed
# They are used by find_echo below for this purpose.
##################################################################
echo_a() { echo      -n "$@";   }
echo_b() { echo         "$@\c"; }
echo_c() { echo      -e "$@\c"; }
echo_d() { /bin/echo -n "$@";   }
echo_e() { /bin/echo    "$@\c"; }
echo_f() { /bin/echo -e "$@\c"; }

##################################################################
# Find out which echo will not append a line feed to the end of
# our string
##################################################################
find_echo()
{
    for i in a b c d e f; do
        test "`echo_$i c``echo_$i a`"  = ca && echon=echo_$i && return
    done
    echon=echo
}

prepare_logs()
{
    if test ! -w $InstallLog; then
        touch "$InstallLog"
    else
        mv $InstallLog $InstallLog.old
        touch "$InstallLog"
    fi
    test -w "$InstallLog" || fatal "Cannot write to file $InstallLog"
    if test ! -w $ErrLog; then
        touch "$ErrLog"
    else
        mv $ErrLog $ErrLog.old
        touch "$ErrLog"
    fi
    test -w "$ErrLog" || fatal "Cannot write to file $ErrLog"

	ts=`date`
    echo "***************************************" >> $InstallLog
    echo "  TIMESTAMP $ts" >> $InstallLog
    echo "***************************************" >> $InstallLog
    echo "***************************************" >> $ErrLog
    echo "  TIMESTAMP $ts" >> $ErrLog
    echo "***************************************" >> $ErrLog
}

##################################################################
# init - performs initialzation for the uninstallation process
# 
##################################################################
init()
{
	
	find_echo
	prepare_logs

	echo "Beginning Uninstallation of Digital's HP OpenView integration" >> $InstallLog
	echo >> $InstallLog
	echo "========== Start INIT Section ==========" >> $InstallLog

	check_for_root
	cd $startdir
	$echon "Initializing..."
	locate_binaries
	check_environment
	echo "Done"

	echo "========== Finish INIT Section ==========" >> $InstallLog
	echo >> $InstallLog
}

##################################################################
# uninstall_mibs
##################################################################
xnmloadmib_status()
{
	case $1 in
		1)	echo "Invalid command line usage" >> $ErrLog;;
		2)  echo "Syntax error in the MIB file $2" >> $ErrLog;;
		4)	echo "MIB name specified by -unload was not loaded" >> $ErrLog;;
		5)  echo "MIB file $2 could not be accessed" >> $ErrLog;;
		6)  echo "MIB file $2 specified with -load is not a file" >> $ErrLog;;
		7)  echo "Error accessing MIB Database" >> $ErrLog;;
		8)  echo "Error creating MIB Database" >> $ErrLog;;
		9)  echo "Error creating compiled form MIB Database" >> $ErrLog;;
		*)  echo "Unknown error" >> $ErrLog;;
	esac
}

uninstall_mibs()
{
	mib_files="RFC1514-HOSTRESOURCES svrsys.mib svrmgt.mib ntcmgt.mib svrntc.mib"

	echo "========== Start UnInstall MIB ==========" >> $InstallLog
	echo >> $InstallLog
#	check_for_files $mib_files
	echo; $echon "UnInstalling MIBs."
	for mf in $mib_files; 
	do
		$OV_BIN/xnmloadmib -unload $mf 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			if test $stat != 4; then
				xnmloadmib_status $stat $mf
			else
				echo "MIB $mf already unloaded" >> $InstallLog
			fi
		fi 
		$echon "."
	done
	$OV_BIN/xnmloadmib -event
	echo "Done" 
	echo "========== Finish UnInstall MIB ==========" >> $InstallLog
	echo >> $InstallLog
}

##################################################################
# uninstall_nt_mib_definitions
##################################################################
edit_oid_to_type()
{
	echo "Editing oid_to_type file" >> $InstallLog
	grep -q "1.3.6.1.4.1.311.1.1.3.1.\*::Windows NT:" oid_to_type
	stat=$?
	if test $stat = 0; then
		sed -e's/1.3.6.1.4.1.311.1.1.3.1.\*::Windows NT:/1.3.6.1.4.1.311.1.1.3.1::Windows NT:/' oid_to_type > oid_to_type.tmp
		mv oid_to_type.tmp oid_to_type
	fi
}

edit_oid_to_sym()
{
	echo "Editing oid_to_sym file" >> $InstallLog

	grep -q "1.3.6.1.4.1.311.1.1.3.1.2:Computer:WindowsNT" oid_to_sym
	stat=$?
	if test $stat = 0; then
		sed -e'/1.3.6.1.4.1.311.1.1.3.1.2:Computer:WindowsNT/d' oid_to_sym > oid_to_sym.tmp
		mv oid_to_sym.tmp oid_to_sym
	fi

	grep -q "1.3.6.1.4.1.311.1.1.3.1.3:Computer:WindowsNT" oid_to_sym
	stat=$?
	if test $stat = 0; then
		sed -e'/1.3.6.1.4.1.311.1.1.3.1.3:Computer:WindowsNT/d' oid_to_sym > oid_to_sym.tmp
		mv oid_to_sym.tmp oid_to_sym
	fi
}

uninstall_nt_mib_definitions()
{
	echo "========== Start UnInstall NT MIB Definitions ==========" >> $InstallLog
	echo >> $InstallLog

	cd $OV_CONF
	check_for_files oid_to_type oid_to_sym

	edit_oid_to_type
	edit_oid_to_sym

	echo "========== Finish UnInstall NT MIB Definitions ==========" >> $InstallLog
	echo >> $InstallLog
	cd $startdir
}
##################################################################
# uninstall_traps
##################################################################
xnmevents_status()
{
	case $1 in
		*)  echo "Error loading trap $2" >> $ErrLog;;
	esac
}

uninstall_traps()
{
	traps="svrThrHighExceptTrap svrThrMediumExceptTr svrThrLowExceptTrap svrThrInformationalE ntcTransitionTrap"

	echo "========== Start UnInstall Traps ==========" >> $InstallLog
	echo >> $InstallLog

	echo; $echon "UnInstalling traps."
	for tf in $traps; 
	do
		$OV_BIN/xnmevents -delete $tf 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			xnmevents_status $stat $tf
		fi
	$echon "."
	done
	echo "Done"
	grep -q "DIGITAL Server Events" $OV_CONF/C/trapd.conf
	stat=$?
	if test $stat = 0; then
		echo "Removing DIGITAL Server Events category from trapd.conf" >> $InstallLog
		sed -e'/DIGITAL Server Events/d' $OV_CONF/C/trapd.conf > $OV_CONF/C/trapd.conf.tmp
		mv $OV_CONF/C/trapd.conf.tmp $OV_CONF/C/trapd.conf 2>> $ErrLog
	fi
	$OV_BIN/xnmevents -event 2>> $ErrLog
	echo
	echo "Removing trap stub files." >> $InstallLog
	if test -w $OV_BIN/DEC/*.sh; then
		$echon "Removing trap stub files."
		for tf in `ls $OV_BIN/DEC/*.sh`;
		do
			rm $tf 2>> $Errlog
			stat=$?
			if test $stat != 0; then
				echo "Unable to remove $tf" >> $ErrLog
				echo "Unable to remove $tf" >> $InstallLog
			fi
			$echon "."
		done
		echo "Done"
	fi
	echo "Removing DECTrap.exe." >> $InstallLog
	rm $OV_BIN/DEC/DECTrap.exe 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		echo "Unable to remove $OV_BIN/DEC/DECTrap.exe" >> $ErrLog
		echo "Unable to remove $OV_BIN/DEC/DECTrap.exe" >> $InstallLog
	fi

	echo "========== Finish UnInstall Traps ==========" >> $InstallLog
	echo >> $InstallLog
}

##################################################################
# uninstall_mib_apps
##################################################################
uninstall_mib_apps()
{

	cd $startdir
	appdir=$OV_REGISTRATION/C/ovmib
	echo "========== Start UnInstall MIB Application Files ==========" >> $InstallLog
	echo >> $InstallLog
	if test -r $appdir; then
		cd $appdir
	else
		 fatal "Unable to find $appdir"
	fi
	
	echo; $echon "UnInstalling MIB Application files."
	for maf in `ls SW*`;
	do
	echo "Removing $OV_REGISTRATION/C/ovmib/$maf" >> $InstallLog
		rm $OV_REGISTRATION/C/ovmib/$maf 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			echo "Unable to remove $OV_REGISTRATION/C/ovmib/$maf" >> $ErrLog
		fi
	$echon "."
	done
	echo "Done"

	cd $startdir
	helpdir=$OV_HELP/C/ovmib/OVW/Functions
	echo "========== Start UnInstall MIB Application Help Files ==========" >> $InstallLog
	echo >> $InstallLog
	if test -r $helpdir; then
		cd $helpdir
	else
		 fatal "Unable to find $helpdir"
	fi
	
	echo; $echon "UnInstalling MIB Application Help files."
	for maf in `ls SW*`;
	do
	echo "Removing $OV_HELP/C/ovmib/OVW/Functions/$maf" >> $InstallLog
		rm $OV_HELP/C/ovmib/OVW/Functions/$maf 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			echo "Unable to remove $OV_HELP/C/ovmib/OVW/Functions/$maf" >> $ErrLog
		fi
	$echon "."
	done
	echo "Done"
	echo "========== Finish UnInstall MIB Application Help Files ==========" >> $InstallLog
	echo >> $InstallLog

	cd $startdir

}

##################################################################
# uninstall_application
##################################################################
uninstall_application()
{
	echo "========== Start UnInstall Application files  ==========" >> $InstallLog
	echo >> $InstallLog
	echo "Removing Application DECDiscover" >> $InstallLog
	rm $OV_BIN/DEC/DECDiscover.exe 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		echo "Unable to remove $OV_BIN/DEC/DECDiscover.exe" >> $ErrLog
	fi

	echo "Removing Application registration file $OV_REGISTRATION/C/DEC/DECDiscover.reg" >> $InstallLog
	rm $OV_REGISTRATION/C/DEC/DECDiscover.reg 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		echo "Unable to remove $OV_REGISTRATION/C/DEC/DECDiscover.reg" >> $ErrLog
	fi

	echo "Removing Field registration file $OV_FIELDS/C/DEC/DECDiscoverFields.reg" >> $InstallLog
	rm $OV_FIELDS/C/DEC/DECDiscoverFields.reg 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		echo "Unable to remove $OV_REGISTRATION/C/DEC/DECDiscover.reg" >> $ErrLog
	fi
	echo "========== Finish UnInstall Application files  ==========" >> $InstallLog
	echo >> $InstallLog
	cd $startdir
}

##################################################################
# uninstall_symbols
##################################################################
uninstall_symbols()
{
	echo "========== Start UnInstall Symbol Files  ==========" >> $InstallLog
	echo >> $InstallLog
	echo "Removing symbol registration file $OV_SYMBOLS/C/DEC/DECSymbols.reg" >> $InstallLog
	rm $OV_SYMBOLS/C/DEC/DECSymbols.reg 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		echo "Unable to remove $OV_SYMBOLS/C/DEC/DECSymbols.reg" >> $ErrLog
	fi

	echo "Removing symbol files $OV_BITMAPS/C/DEC/*" >> $InstallLog
	rm $OV_BITMAPS/C/DEC/* 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		echo "Unable to remove $OV_BITMAPS/C/DEC/*" >> $ErrLog
	fi

	echo "========== Finish UnInstall Symbol Files  ==========" >> $InstallLog
	echo >> $InstallLog
}
##################################################################
# uninstall_uninstall
##################################################################
uninstall_uninstall()
{
	echo "========== Start UnInstall Uninstall ==========" >> $InstallLog
	echo >> $InstallLog
	rm $OV_BIN/DEC/uninstall.sh 2>> $ErrLog
	stat=$?
	if test $stat != 0; then
		echo "Unable to remove $OV_BIN/DEC/uninstall.sh" >> $ErrLog
	fi
	echo "========== Finish UnInstall Uninstall  ==========" >> $InstallLog
	echo >> $InstallLog
}

##################################################################
# remove_directories
##################################################################
remove_directories ()
{
	dirs="$OV_BIN/DEC $OV_BITMAPS/C/DEC $OV_SYMBOLS/C/DEC $OV_FIELDS/C/DEC $OV_REGISTRATION/C/DEC"
	echo "========== Start Remove Directories  ==========" >> $InstallLog
	echo >> $InstallLog
	echo
	echo
	$echon "Removing directories."
	for d in $dirs;
	do
		echo "Removing directory $d" >> $InstallLog
		rmdir $d 2>> $ErrLog
		stat=$?
		if test $stat != 0; then
			echo "Unable to remove directory $d" >> $ErrLog
		fi	
	$echon "."
	done
	echo "Done"
	echo "========== Finish Remove Directories  ==========" >> $InstallLog
	echo >> $InstallLog
	echo
}
##################################################################
# Welcome
##################################################################
welcome()
{
	echo 
	echo
	echo
	echo
	echo
	echo
	echo "		Digital ServerWORKS Integration for OpenView"
	echo "		(c) Copyright, 1998 Digital Equipment Corporation"
	echo
	echo
	echo
	echo
	echo
	readln
}

##################################################################
# cleanup
##################################################################
cleanup()
{
	echo "========== UnInstallation Ends ==========" >> $InstallLog
	echo >> $InstallLog
	echo; echo;
	if yesno 'Do you want to view the Uninstalltion log'; then
		clear
		cat $InstallLog | more
		echo
		readln
	fi
	if (test -f "$InstallLog") then
		if yesno 'Do you want to delete the uninstall log'; then
			rm "$InstallLog"
		fi
	fi
	echo
	echo
	if yesno 'Do you want to view the error log'; then
		clear
		cat $ErrLog | more
		echo
		readln
	fi
	if (test -f "$ErrLog") then
		if yesno 'Do you want to delete the error log'; then
			rm "$ErrLog"
		fi
	fi
}

##################################################################
# main()
##################################################################

clear
init
clear
welcome
clear
echo
uninstall_mibs
uninstall_nt_mib_definitions
uninstall_traps
uninstall_mib_apps
uninstall_application
uninstall_symbols
uninstall_uninstall
remove_directories
echo
echo "Uninstallation Completed!"
echo
cleanup

