#!/bin/sh
#
# Install script for Digital extensions to HP OpenView
# (c) Copyright 1998, Digital Equipment Corporation
#
##################################################################

##################################################################
# global variables
##################################################################
startdir=`pwd`
kitdir="/tmp/DECHPOV"

##################################################################
# Error handlers
##################################################################
fatal()
{
	echo; echo
	echo "ERROR: $@."
	echo "This was a fatal error.  Installation aborted."
	echo 
	cleanup
	exit 1
}

##################################################################
# User interaction routines
##################################################################
yesno()
{
	while true; do
	$echon "$1 (Y/N)? "
	read ans
	case $ans in
		y|Y)	return 0;;
		n|N)	return 1;;
	esac
	done
}

readln()
{
	$echon 'Press return to continue... '
	read jnk
}

##################################################################
# The routines that follow are all used to make sure that we
# have the correct environment for installation.  Are we root,
# are the tools we need installed, has the OV environment 
# been set.
##################################################################

check_for_root()
{
	echo "Checking that we are super-user" 
	if [ "`whoami`" != "root" ] ; then
		fatal "Installation must be performed as the super-user!"
	fi
}

check_environment()
{
	echo "Checking that the OpenView environment is set" 
	test -z "$OV_BIN" && fatal "OpenView environment not set.  Execute ov.envvars.sh and try again"
}

check_for_file()
{
	echo "Checking for file $1" 
	test -r "$1" && return 0
	return 1
}

check_for_files()
{
    for this_bin
    do
        check_for_file $this_bin ||
          fatal "file '$this_bin' not found"
    done
}

check_for_binary()
{
echo "Checking for binary $1" 
    testbin=$1
    case "$testbin" in
      /*) test -x "$testbin" && test -f "$testbin"; return;;
      *)
         OLDIFS=$IFS; IFS=:; eval set $PATH; IFS=$OLDIFS
         for this_dir
         do
             test -x "$this_dir/$testbin" &&
             test -f "$this_dir/$testbin" && return 0
         done
         return 1;;
    esac
}

require_binaries()
{
    for this_bin
    do
        check_for_binary $this_bin ||
          fatal "program '$this_bin' not found in PATH"
    done
}

locate_binaries()
{
	require_binaries bdf touch sed awk ln rm ls tr mkdir cat pwd tar compress
}

check_for_directory()
{
echo "Checking for directory $1" 
	test -r "$1" && return 0
	return 1
}

make_directory()
{
	check_for_directory $1 && return
	echo "Making directory $1" 
	mkdir $1 && return
	fatal "Unable to make directory $1"
}

check_disk_space()
{
	echo "Checking disk space"
	availfree=`bdf "$kitdir" | awk 'NR == 2 {print $4}'`
	if [ $availfree -lt 28000 ]; then
		fatal "Not enough disk space.  28000 512-byte blocks required. $availfree available"
	fi
}

##################################################################
# Here are a set of test functions for echo.  The purpose is to
# find which one will do the echo without appending a line feed
# They are used by find_echo below for this purpose.
##################################################################
echo_a() { echo      -n "$@";   }
echo_b() { echo         "$@\c"; }
echo_c() { echo      -e "$@\c"; }
echo_d() { /bin/echo -n "$@";   }
echo_e() { /bin/echo    "$@\c"; }
echo_f() { /bin/echo -e "$@\c"; }

##################################################################
# Find out which echo will not append a line feed to the end of
# our string
##################################################################
find_echo()
{
    for i in a b c d e f; do
        test "`echo_$i c``echo_$i a`"  = ca && echon=echo_$i && return
    done
    echon=echo
}

##################################################################
# init - performs initialzation for the installation process
# 
##################################################################
init()
{
	
	find_echo
	echo
	echo "========== Installation Begins =========="
	echo
	echo "Initializing."
	check_for_root
	cd $startdir
	locate_binaries
	check_environment
	make_directory $kitdir
	check_disk_space
	echo "Done"
	echo
}

##################################################################
# unload_kit
##################################################################
unload_kit()
{
	kitfiles=DECHPOV.tar
	check_for_files $kitfiles
	make_directory $kitdir
	echo
	echo "========== Unpacking =========="
	echo
	cd $kitdir
	for tf in $kitfiles; 
	do
		echo "Unpacking $tf to $kitdir temporarily"
		tar xf $startdir/$tf 
		stat=$?
		if test $stat != 0; then
			fatal "Error unpacking $tf"
		fi
	done
	echo 
	echo "========== Unpacking Complete =========="
	echo
	cd $startdir
}

##################################################################
# start_install
##################################################################
start_install()
{
	cd $kitdir/OpenView
    echo
    echo
    echo "Kit for Digital ServerWORKS Integration for OpenView unpacked"
    echo "Installation ready to begin"
    echo 
    readln
	./install.sh
	cd $startdir
}
##################################################################
# Welcome
##################################################################
welcome()
{
        echo
        echo
        echo
        echo
        echo
        echo
		echo "		Preparing to install "
        echo "          Digital ServerWORKS Integration for OpenView"
        echo
        echo
        echo
        echo
        echo
        readln
}
##################################################################
# cleanup
##################################################################
cleanup()
{
	
	if test -r $kitdir; then
		echo
		echo "Removing temporary kit $kitdir"
		echo
		rm -rf $kitdir
	fi
	echo "========== Installation Ends =========="
}

##################################################################
# main()
##################################################################

clear
init
welcome
clear
echo
unload_kit
start_install
cleanup

