#include "snd.h"

#if USE_MOTIF
  #define BIG_DOT_SIZE 10
  #define MEDIUM_DOT_SIZE 7
  #define LITTLE_DOT_SIZE 4
#else
  #define BIG_DOT_SIZE 5
  #define MEDIUM_DOT_SIZE 3
  #define LITTLE_DOT_SIZE 2
#endif

static int current_dot_size = BIG_DOT_SIZE;

#define MAX_PIXEL 10000


env *free_env(env *e)
{
  if (e)
    {
      if (e->data) {free(e->data); e->data = NULL;}
      free(e);
    }
  return(NULL);
}


env *copy_env(env *e)
{
  if (e)
    {
      env *ne;
      ne = (env *)calloc(1, sizeof(env));
      ne->pts = e->pts;
      ne->data_size = e->pts * 2;
      ne->data = (mus_float_t *)malloc(ne->data_size * sizeof(mus_float_t));
      memcpy((void *)(ne->data), (void *)(e->data), ne->data_size * sizeof(mus_float_t));
      ne->base = e->base;
      return(ne);
    }
  return(NULL);
}


bool envs_equal(env *e1, env *e2)
{
  /* snd-mix.c check for set mix amp env no-op */
  int i;
  if (!e1) return(!e2);
  if (!e2) return(false);
  if (e1->pts != e2->pts) return(false);
  for (i = 0; i < e1->pts * 2; i++)
    if (e1->data[i] != e2->data[i])
      return(false);
  if (e1->base != e2->base) return(false); /* 1 and 0 are possibilities here */
  return(true);
}


char *env_to_string(env *e)
{
  char *news = NULL;
  if (e)
    {
      int i, j, len;
      bool first = true;
      char *expr_buf;
      len = 4 + (e->pts * 2 * 16);
      news = (char *)calloc(len, sizeof(char));
#if HAVE_RUBY
      news[0] = '[';
#endif
#if HAVE_SCHEME
      news[0] = '\'';
      news[1] = '(';
#endif
#if HAVE_FORTH
      news[0] = '\'';
      news[1] = '(';
      news[2] = ' ';
#endif
      expr_buf = (char *)calloc(PRINT_BUFFER_SIZE, sizeof(char));
      for (i = 0, j = 0; i < e->pts; i++, j += 2)
	{
	  if (fabs(e->data[j + 1]) < .0000001) e->data[j + 1] = 0.0; /* try to get rid of -0.000 */
#if HAVE_RUBY
	  mus_snprintf(expr_buf, PRINT_BUFFER_SIZE, "%s%.3f, %.3f", (first) ? "" : ", ", e->data[j], e->data[j + 1]);
#endif
#if HAVE_SCHEME || HAVE_FORTH
	  mus_snprintf(expr_buf, PRINT_BUFFER_SIZE, "%s%.3f %.3f", (first) ? "" : " ", e->data[j], e->data[j + 1]);
#endif
	  news = mus_strcat(news, expr_buf, &len);
	  first = false;
	}
      free(expr_buf);
#if HAVE_RUBY
      news = mus_strcat(news, "]", &len);
#endif
#if HAVE_SCHEME
      news = mus_strcat(news, ")", &len);
#endif
#if HAVE_FORTH
      news = mus_strcat(news, " )", &len);
#endif
    }
  else news = mus_strdup(PROC_FALSE);
  return(news);
}


env *make_envelope_with_offset_and_scaler(mus_float_t *env_buffer, int len, mus_float_t offset, mus_float_t scaler)
{
  env *e;
  int i, flen;
  if (len < 4) flen = 4; else flen = len;
  if (flen & 1) flen++;
  e = (env *)calloc(1, sizeof(env));
  e->data = (mus_float_t *)calloc(flen, sizeof(mus_float_t));
  e->data_size = flen;
  e->pts = flen / 2;
  for (i = 0; i < len; i += 2) 
    {
      e->data[i] = env_buffer[i];
      e->data[i + 1] = offset + scaler * env_buffer[i + 1];
    }
  if ((flen == 4) && (len == 2))  /* fixup degenerate envelope */
    {
      e->data[2] = e->data[0] + 1.0; 
      e->data[3] = e->data[1];
    }
  e->base = 1.0;
  return(e);
}


static env *make_envelope(mus_float_t *env_buffer, int len)
{
  return(make_envelope_with_offset_and_scaler(env_buffer, len, 0.0, 1.0));
}


static void add_point(env *e, int pos, mus_float_t x, mus_float_t y)
{
  int i, j;
  if (e->pts * 2 == e->data_size)
    {
      e->data_size += 16;
      e->data = (mus_float_t *)realloc(e->data, (e->data_size) * sizeof(mus_float_t));
    }
  for (i = e->pts - 1, j = (e->pts - 1) * 2; i >= pos; i--, j -= 2)
    {
      e->data[j + 2] = e->data[j];
      e->data[j + 3] = e->data[j + 1];
    }
  e->data[pos * 2] = x;
  e->data[pos * 2 + 1] = y;
  e->pts++;
}


static void move_point(env *e, int pos, mus_float_t x, mus_float_t y)
{
  e->data[pos * 2] = x;
  e->data[pos * 2 + 1] = y;
}


static void delete_point(env *e, int pos)
{
  int i, j;
  for (i = pos, j = pos * 2; i < e->pts - 1; i++, j += 2)
    {
      e->data[j] = e->data[j + 2];
      e->data[j + 1] = e->data[j + 3];
    }
  e->pts--;
}


static int place_point(int *cxs, int points, int x, env *e, mus_float_t bx)
{
  int i;
  for (i = 0; i < points; i++)
    {
      if (x == cxs[i])
	{
	  /* use true values to disambiguate */
	  if (e->data[i * 2] > bx)
	    return(i - 1);
	  else return(i);
	}
      else
	{
	  if (x < cxs[i]) 
	    return(i - 1);
	}
    }
  return(points);
}


static int hit_point(int *cxs, int *cys, int points, int x, int y)
{
  /* enved dot size (10) is big enough that we need to search for the closest dot
   *   but I think we can assume that the x's are in order
   */
  int i, cur_i = -1, cur_min_x = MAX_PIXEL, cur_min_y = MAX_PIXEL, lim_x;
  lim_x = x + current_dot_size;
  for (i = 0; (i < points) && (cxs[i] <= lim_x); i++)
    if (((x > (cxs[i] - current_dot_size)) && 
	 (x < (cxs[i] + current_dot_size))) &&
	((y > (cys[i] - current_dot_size)) && 
	 (y < (cys[i] + current_dot_size))))
      {
	if (abs(x - cxs[i]) <= cur_min_x)
	  {
	    if (abs(y - cys[i]) < cur_min_y)
	      {
		cur_i = i;
		cur_min_x = abs(x - cxs[i]);
		cur_min_y = abs(y - cys[i]);
	      }
	  }
      }
  return(cur_i);
}


env *default_env(mus_float_t x1, mus_float_t y)
{
  env *e;
  e = (env *)calloc(1, sizeof(env));
  e->data = (mus_float_t *)calloc(4, sizeof(mus_float_t));
  e->data_size = 4;
  e->pts = 2;
  e->data[0] = 0.0; 
  e->data[1] = y; 
  e->data[2] = x1;
  e->data[3] = y;
  e->base = 1.0;
  return(e);
}


bool default_env_p(env *e)
{
  if (e == NULL) return(true);
  if (e->pts != 2) return(false);
  return((snd_feq(e->data[0], 0.0)) &&
	 (snd_feq(e->data[1], 1.0)) &&
	 (snd_feq(e->data[2], 1.0)) &&
	 (snd_feq(e->data[3], 1.0)));
}


env_editor *new_env_editor(void)
{
  env_editor *edp;
  edp = (env_editor *)calloc(1, sizeof(env_editor));
  edp->current_xs = (int *)calloc(8, sizeof(int));
  edp->current_ys = (int *)calloc(8, sizeof(int));
  edp->axis = (axis_info *)calloc(1, sizeof(axis_info));
  edp->current_size = 8;
  edp->env_dragged = false;
  edp->env_pos = 0;
  edp->click_to_delete = false;
  edp->edited = false;
  edp->clip_p = true;
  edp->in_dB = false;
  return(edp);
}


static void env_editor_set_current_point(env_editor *edp, int pos, int x, int y)
{
  if (pos == edp->current_size)
    {
      edp->current_size += 8;
      edp->current_xs = (int *)realloc(edp->current_xs, edp->current_size * sizeof(int));
      edp->current_ys = (int *)realloc(edp->current_ys, edp->current_size * sizeof(int));
    }
  edp->current_xs[pos] = x;
  edp->current_ys[pos] = y;
}


static short env_editor_grf_y_dB(env_editor *edp, mus_float_t val)
{
  if (edp->in_dB)
    return(grf_y(in_dB(min_dB(ss), ss->lin_dB, val), edp->axis));
  else return(grf_y(val, edp->axis));
}


static mus_float_t un_dB(mus_float_t py)
{
  return((py <= min_dB(ss)) ? 0.0 : pow(10.0, py * .05));
}


double env_editor_ungrf_y_dB(env_editor *edp, int y)
{
  if (edp->in_dB)
    return(un_dB(ungrf_y(edp->axis, y)));
  else return(ungrf_y(edp->axis, y));
}


#define EXP_SEGLEN 4
typedef enum {ENVED_ADD_POINT, ENVED_DELETE_POINT, ENVED_MOVE_POINT} enved_point_t;

static bool check_enved_hook(env *e, int pos, mus_float_t x, mus_float_t y, enved_point_t reason);

/* enved display can call mus_make_env which can throw 'mus-error, so we need local protection */
static mus_error_handler_t *old_error_handler;

static void local_mus_error(int type, char *msg)
{
  snd_error_without_format(msg);
}


void env_editor_display_env(env_editor *edp, env *e, graphics_context *ax, const char *name, 
			    int x0, int y0, int width, int height, printing_t printing)
{
  int i, j, k;
  mus_float_t ex0, ey0, ex1, ey1, val;
  int ix0, ix1, iy0, iy1, lx0, lx1, ly0, ly1, index = 0;
  mus_float_t env_val, curx, xincr;
  int dur;
  axis_info *ap;
  if (e)
    {
      ex0 = e->data[0];
      ey0 = e->data[1];
      ex1 = e->data[(e->pts * 2) - 2];
      ey1 = ey0;
      for (i = 3; i < e->pts * 2; i += 2)
	{
	  val = e->data[i];
	  if (ey0 > val) ey0 = val;
	  if (ey1 < val) ey1 = val;
	}
      if (ey0 > 0.0) ey0 = 0.0;
      if ((ey0 == ey1) && (ey1 == 0.0)) ey1 = 1.0; /* fixup degenerate case */
      if ((edp->with_dots) && (ey1 < 1.0)) ey1 = 1.0;
    }
  else
    {
      if (edp != ss->enved) return;
      ex0 = 0.0;
      ex1 = 1.0;
      ey0 = 0.0;
      ey1 = 1.0;
    }
  if (edp->in_dB)
    {
      ey0 = min_dB(ss); 
      ey1 = 0.0;
    }
  if (edp == ss->enved)
    {
      /* don't free edp->axis! */
      ap = enved_make_axis(name, ax, x0, y0, width, height, ex0, ex1, ey0, ey1, printing); /* ax used only for GC here */
      edp->axis = ap;
    }
  else 
    {
      ap = edp->axis;
      ap->ax = ax;
      if (edp->with_dots)
	init_env_axes(ap, name, x0, y0, width, height, ex0, ex1, ey0, ey1, NOT_PRINTING);
    }
  if (!(ax->wn)) return;

  if (e)
    {
      ix1 = grf_x(e->data[0], ap);
      iy1 = env_editor_grf_y_dB(edp, e->data[1]);
      if (edp->with_dots)
	{
	  if (e->pts < 50)
	    current_dot_size = BIG_DOT_SIZE;
	  else
	    {
	      if (e->pts < 100)
		current_dot_size = MEDIUM_DOT_SIZE;
	      else current_dot_size = LITTLE_DOT_SIZE;
	    }
	  env_editor_set_current_point(edp, 0, ix1, iy1);
	  draw_dot(ax, ix1, iy1, current_dot_size);
	}
      if (e->base == 1.0)
	{
	  if (edp->in_dB)
	    {
	      for (j = 1, i = 2; i < e->pts * 2; i += 2, j++)
		{
		  ix0 = ix1;
		  iy0 = iy1;
		  ix1 = grf_x(e->data[i], ap);
		  iy1 = env_editor_grf_y_dB(edp, e->data[i + 1]);
		  if (edp->with_dots)
		    {
		      env_editor_set_current_point(edp, j, ix1, iy1);
		      draw_dot(ax, ix1, iy1, current_dot_size);
		    }
		  /* now try to fill in from the last point to this one */
		  if ((ix1 - ix0) < (2 * EXP_SEGLEN))
		    {
		      /* points are too close to be worth interpolating */
		      draw_line(ax, ix0, iy0, ix1, iy1);
		    }
		  else
		    {
		      /* interpolate so the display looks closer to dB */
		      mus_float_t yval, yincr;
		      dur = (ix1 - ix0) / EXP_SEGLEN;
		      xincr = (e->data[i] - e->data[i - 2]) / (mus_float_t)dur;
		      curx = e->data[i - 2] + xincr;
		      lx1 = ix0;
		      ly1 = iy0;
		      yval = e->data[i - 1];
		      yincr = (e->data[i + 1] - yval) / (mus_float_t)dur;
		      yval += yincr;
		      for (k = 1; k < dur; k++, curx += xincr, yval += yincr)
			{
			  lx0 = lx1;
			  ly0 = ly1;
			  lx1 = grf_x(curx, ap);
			  ly1 = grf_y(in_dB(min_dB(ss), ss->lin_dB, yval), ap);
			  draw_line(ax, lx0, ly0, lx1, ly1);
			}
		      draw_line(ax, lx1, ly1, ix1, iy1);
		    }
		}
	    }
	  else
	    {
	      for (j = 1, i = 2; i < e->pts * 2; i += 2, j++)
		{
		  ix0 = ix1;
		  iy0 = iy1;
		  ix1 = grf_x(e->data[i], ap);
		  iy1 = grf_y(e->data[i + 1], ap);
		  if (edp->with_dots)
		    {
		      env_editor_set_current_point(edp, j, ix1, iy1);
		      draw_dot(ax, ix1, iy1, current_dot_size);
		    }
		  draw_line(ax, ix0, iy0, ix1, iy1);
		  if (printing) ps_draw_line(ap, ix0, iy0, ix1, iy1);
		}
	    }
	}
      else
	{
	  if (e->base <= 0.0)
	    {
	      for (j = 1, i = 2; i < e->pts * 2; i += 2, j++)
		{
		  ix0 = ix1;
		  iy0 = iy1;
		  ix1 = grf_x(e->data[i], ap);
		  iy1 = env_editor_grf_y_dB(edp, e->data[i + 1]);
		  if (edp->with_dots)
		    {
		      env_editor_set_current_point(edp, j, ix1, iy1);
		      draw_dot(ax, ix1, iy1, current_dot_size);
		    }
		  draw_line(ax, ix0, iy0, ix1, iy0);
		  draw_line(ax, ix1, iy0, ix1, iy1);
		  if (printing) 
		    {
		      ps_draw_line(ap, ix0, iy0, ix1, iy0);
		      ps_draw_line(ap, ix1, iy0, ix1, iy1);
		    }
		}
	    }
	  else
	    {
	      mus_any *ce;
	      if (edp->with_dots)
		for (j = 1, i = 2; i < e->pts * 2; i += 2, j++)
		  env_editor_set_current_point(edp, j, grf_x(e->data[i], ap), grf_y(e->data[i + 1], ap));

	      /* exponential case */
	      dur = width / EXP_SEGLEN;
	      old_error_handler = mus_error_set_handler(local_mus_error);
	      ce = mus_make_env_with_length(e->data, e->pts, 1.0, 0.0, e->base, dur);
	      mus_error_set_handler(old_error_handler);
	      if (ce == NULL) return;
	      if (dur < e->pts) dur = e->pts;
	      env_val = mus_env(ce);
	      ix1 = grf_x(0.0, ap);
	      iy1 = env_editor_grf_y_dB(edp, env_val);
	      xincr = (ex1 - ex0) / (mus_float_t)dur;
	      for (i = 1, curx = ex0 + xincr; i < dur; i++, curx += xincr)
		{
		  iy0 = iy1;
		  ix0 = ix1;
		  env_val = mus_env(ce);
		  ix1 = grf_x(curx, ap);
		  iy1 = env_editor_grf_y_dB(edp, env_val);
		  draw_line(ax, ix0, iy0, ix1, iy1);
		  if (printing) ps_draw_line(ap, ix0, iy0, ix1, iy1);
		  if ((edp->with_dots) && (index != mus_position(ce)))
		    {
		      index = mus_position(ce);
		      if (index < (e->pts - 1))
			draw_dot(ax, ix1, iy1, current_dot_size);
		    }
		}
	      if (curx < ex1)
		{
		  iy0 = iy1;
		  ix0 = ix1;
		  ix1 = grf_x(ex1, ap);
		  iy1 = env_editor_grf_y_dB(edp, e->data[e->pts * 2 - 1]);
		  draw_line(ax, ix0, iy0, ix1, iy1);
		  if (printing) ps_draw_line(ap, ix0, iy0, ix1, iy1);
		}
	      if (edp->with_dots)
		draw_dot(ax, ix1, iy1, current_dot_size);
	      mus_free(ce);
	    }
	}
    }
}


void env_editor_button_motion_with_xy(env_editor *edp, int evx, int evy, oclock_t motion_time, env *e, mus_float_t *new_x, mus_float_t *new_y)
{
  axis_info *ap;
  mus_float_t x0, x1, x, y;
  if ((e == NULL) || (edp == NULL)) return;
  if ((motion_time - edp->down_time) < ss->click_time) return;
  edp->env_dragged = true;
  edp->click_to_delete = false;
  ap = edp->axis;
  x = ungrf_x(ap, evx);
  if (edp->env_pos > 0) 
    x0 = e->data[edp->env_pos * 2 - 2]; 
  else x0 = e->data[0];
  if (edp->env_pos < (e->pts - 1))
    x1 = e->data[edp->env_pos * 2 + 2]; /* looking for next point on right to avoid crossing it */
  else x1 = e->data[e->pts * 2 - 2];
  {
    mus_float_t dist = 0.0001;
    if ((x1 - x0) <= dist)
      dist = (x1 - x0) / 2.0;
    if (x <= x0) x = x0 + dist;
    if (x >= x1) x = x1 - dist;
  }
  if (edp->env_pos == 0) x = e->data[0];
  if (edp->env_pos == (e->pts - 1)) x = e->data[(e->pts - 1) * 2];
  y = ungrf_y(ap, evy);
  if ((edp->clip_p) || (edp->in_dB))
    {
      if (y < ap->y0) y = ap->y0;
      if (y > ap->y1) y = ap->y1;
    }
  if (edp->in_dB) y = un_dB(y);
  if ((edp != ss->enved) || 
      (check_enved_hook(e, edp->env_pos, x, y, ENVED_MOVE_POINT) == 0))
    move_point(e, edp->env_pos, x, y);
  edp->edited = true;
  if (new_x) (*new_x) = x;
  if (new_y) (*new_y) = y;
}


void env_editor_button_motion(env_editor *edp, int evx, int evy, oclock_t motion_time, env *e)
{
  env_editor_button_motion_with_xy(edp, evx, evy, motion_time, e, NULL, NULL);
}


bool env_editor_button_press(env_editor *edp, int evx, int evy, oclock_t time, env *e)
{
  int pos;
  mus_float_t x, y;
  axis_info *ap;
  ap = edp->axis;
  edp->down_time = time;
  edp->env_dragged = false;
  pos = hit_point(edp->current_xs, edp->current_ys, e->pts, evx, evy);
  x = ungrf_x(ap, evx);
  y = env_editor_ungrf_y_dB(edp, evy);
  if (edp->clip_p)
    {
      if (y < ap->y0) y = ap->y0;
      if (y > ap->y1) y = ap->y1;
    }
  if (pos == -1)
    {
      if (x <= ap->x0)
	{
	  pos = 0;
	  x = ap->x0;
	}
      else 
	{
	if (x >= ap->x1) 
	  {
	    pos = e->pts - 1;
	    x = ap->x1;
	  }
	}
    }
  edp->env_pos = pos;
  /* if not -1, then user clicked existing point -- wait for drag/release to decide what to do */
  if (pos == -1) 
    {
      pos = place_point(edp->current_xs, e->pts, evx, e, x);
      /* place returns left point index of current segment or pts if off left end */
      /* in this case, user clicked in middle of segment, so add point there */
      if ((edp != ss->enved) || (check_enved_hook(e, pos, x, y, ENVED_ADD_POINT) == 0))
	add_point(e, pos + 1, x, y);
      edp->env_pos = pos + 1;
      edp->click_to_delete = false;
    }
  else edp->click_to_delete = true;
  edp->edited = true;
  return(pos == -1);
}


void env_editor_button_release(env_editor *edp, env *e)
{
  if ((edp->click_to_delete) && 
      (!(edp->env_dragged)) && 
      (edp->env_pos > 0) && 
      (edp->env_pos < (e->pts - 1)) &&
      ((edp != ss->enved) || (check_enved_hook(e, edp->env_pos, 0, 0, ENVED_DELETE_POINT) == 0)))
    delete_point(e, edp->env_pos);
  prepare_enved_edit(e);
  edp->env_pos = 0;
  edp->env_dragged = false;
  edp->click_to_delete = false;
}



/* -------- (main) ENVELOPE EDITOR FUNCTIONS -------- */

static int env_list_size = 0;    /* current size of env edits list */
static int env_list_top = 0;     /* one past current active position in list */
static env **env_list = NULL;    /* env edits list (for local undo/redo/revert) */

static env **all_envs = NULL;    /* all envs, either loaded or created in editor */
static char **all_names = NULL;  /* parallel names */
static int all_envs_size = 0;    /* size of this array */
static int all_envs_top = 0;     /* one past pointer to last entry in this array */


void init_env_axes(axis_info *ap, const char *name, int x_offset, int ey0, int width, int height, 
		   mus_float_t xmin, mus_float_t xmax, mus_float_t ymin, mus_float_t ymax, printing_t printing)
{
  if (ap->xlabel) free(ap->xlabel);
  ap->xmin = xmin;
  ap->xmax = xmax;
  ap->ymin = ymin;
  ap->ymax = ymax;
  ap->y_ambit = ap->ymax - ap->ymin;
  ap->x_ambit = ap->xmax - ap->xmin;
  ap->xlabel = mus_strdup(name);
  ap->x0 = xmin;
  ap->x1 = xmax;
  ap->y0 = ymin;
  ap->y1 = ymax;
  ap->width = width;
  ap->window_width = width;
  ap->y_offset = ey0;
  ap->height = height;
  ap->graph_x0 = x_offset;
  make_axes_1(ap, X_AXIS_IN_SECONDS, 1, SHOW_ALL_AXES, printing, WITH_X_AXIS, NO_GRID, WITH_LINEAR_AXES, grid_density(ss));
  /* if this is too small for an axis, it still sets up the fields needed for grf_x|y, so tiny envelope graphs will work */
}


void view_envs(int env_window_width, int env_window_height, printing_t printing)
{
  /* divide space available into a grid (if needed) that shows all currently defined envelopes */
  /* I suppose if there were several hundred envelopes, we'd need a scrollable viewer... */
  int cols, rows, i, j, width, height, x, y, k;
  if (all_envs_top > 1)
    {
      cols = snd_round(sqrt((mus_float_t)(all_envs_top * env_window_width) / (mus_float_t)env_window_height));
      rows = snd_round((mus_float_t)all_envs_top / (mus_float_t)cols);
      if ((rows * cols) < all_envs_top) rows++;
    }
  else
    {
      cols = 1;
      rows = 1;
    }
  width = (int)((mus_float_t)env_window_width / (mus_float_t)cols);
  height = (int)((mus_float_t)env_window_height / (mus_float_t)rows);
  k = 0;
  for (i = 0, x = 0; i < cols; i++, x += width)
    for (j = 0, y = 0; j < rows; j++, y += height)
      {
	display_enved_env_with_selection(all_envs[k], all_names[k], x, y, width, height, 0, printing);
	k++;
	if (k == all_envs_top) return;
      }
}


int hit_env(int xe, int ye, int env_window_width, int env_window_height)
{
  if (all_envs_top == 0)
    return(-1);
  else
    {
      if (all_envs_top == 1)
	return(0);
      else
	{
	  int cols, rows, i, j, width, height, x, y, k;
	  cols = snd_round(sqrt((mus_float_t)(all_envs_top * env_window_width) / (mus_float_t)env_window_height));
	  rows = snd_round((mus_float_t)all_envs_top / (mus_float_t)cols);
	  if ((rows * cols) < all_envs_top) rows++;
	  width = (int)((mus_float_t)env_window_width / (mus_float_t)cols);
	  height = (int)((mus_float_t)env_window_height / (mus_float_t)rows);
	  k = 0;
	  for (i = 0, x = width; i < cols; i++, x += width)
	    if (x > xe)
	      for (j = 0, y = height; j < rows; j++, y += height)
		{
		  if (y > ye) return(k);
		  k++;
		}
	    else k += rows;
	}
    }
  return(0);
}


void prepare_enved_edit(env *new_env)
{
  int i;
  if (env_list_top == env_list_size)
    {
      env_list_size += 16;
      if (env_list)
	{
	  env_list = (env **)realloc(env_list, env_list_size * sizeof(env *));
	  for (i = env_list_top; i < env_list_size; i++) env_list[i] = NULL;
	}
      else env_list = (env **)calloc(env_list_size, sizeof(env *));
    }
  /* clear out current edit list above this edit */
  for (i = env_list_top; i < env_list_size; i++)
    env_list[i] = free_env(env_list[i]);
  env_list[env_list_top] = copy_env(new_env);
  env_list_top++;
}


void redo_env_edit(void)
{
  if (env_list)
    {
      if ((env_list_top < env_list_size) && 
	  (env_list[env_list_top])) 
	{
	  env_list_top++;
	  set_enved_undo_sensitive(true);
	  set_enved_revert_sensitive(true);
	}
      if ((env_list_top == env_list_size) || 
	  (env_list[env_list_top] == NULL)) 
	set_enved_redo_sensitive(false);
      set_enved_save_sensitive(true);
    }
}


void undo_env_edit(void)
{
  if (env_list)
    {
      if (env_list_top > 0)
	{
	  env_list_top--;
	  set_enved_redo_sensitive(true);
	}
      if (env_list_top == 0)
	{
	  set_enved_undo_sensitive(false);
	  /* set_enved_revert_sensitive(false); */
	}
      set_enved_save_sensitive(true);
    }
}


void revert_env_edit(void)
{
  if (env_list)
    {
      if (env_list_top > 0)
	set_enved_redo_sensitive(true);
      if (env_list_top > 1) 
	env_list_top = 1; 
      else 
	{
	  env_list_top = 0;
	  set_enved_undo_sensitive(false);
	  set_enved_revert_sensitive(false);
	  set_enved_save_sensitive(false);
	}
    }
}


static int find_env(const char *name)
{ /* -1 upon failure */
  int i;
  if ((all_envs) && 
      (name))
    for (i = 0; i < all_envs_top; i++)
      if (mus_strcmp(name, all_names[i]))
	return(i);
  return(-1);
}


int enved_all_envs_top(void) {return(all_envs_top);}
char *enved_all_names(int n) {return(all_names[n]);}
void set_enved_env_list_top(int n) {env_list_top = n;}
env *enved_all_envs(int pos) {return(all_envs[pos]);}


static void add_envelope(const char *name, env *val)
{
  if (all_envs_top == all_envs_size)
    {
      all_envs_size += 16;
      if (all_envs)
	{
	  int i;
	  all_envs = (env **)realloc(all_envs, all_envs_size * sizeof(env *));
	  all_names = (char **)realloc(all_names, all_envs_size * sizeof(char *));
	  for (i = all_envs_size - 16; i < all_envs_size; i++) {all_names[i] = NULL; all_envs[i] = NULL;}
	}
      else
	{
	  all_envs = (env **)calloc(all_envs_size, sizeof(env *));
	  all_names = (char **)calloc(all_envs_size, sizeof(char *));
	}
    }
  all_envs[all_envs_top] = val;
  if (all_names[all_envs_top]) free(all_names[all_envs_top]);
  all_names[all_envs_top] = mus_strdup(name);
  all_envs_top++;
  if (enved_dialog_is_active())
    {
      set_enved_show_sensitive(true);
      make_scrolled_env_list();
    }
}


void delete_envelope(const char *name)
{
  int pos;
  pos = find_env(name);
  if (pos != -1)
    {
      int i;
      if (all_names[pos]) free(all_names[pos]);
      for (i = pos; i < all_envs_size - 1; i++)
	{
	  all_envs[i] = all_envs[i + 1]; 
	  all_envs[i + 1] = NULL;
	  all_names[i] = all_names[i + 1]; 
	  all_names[i + 1] = NULL;
	}
      all_envs_top--;
      if (enved_dialog_is_active())
	{
	  if (all_envs_top > 0)
	    set_enved_show_sensitive(true);
	  make_scrolled_env_list();
	}
    }
}


void alert_envelope_editor(const char *name, env *val)
{
  /* whenever an envelope is defined, we get notification through this function */
  int i;
  if (val == NULL) return;
  i = find_env(name);
  if (i != -1)
    {
      free_env(all_envs[i]);
      all_envs[i] = val;
    }
  else add_envelope(name, val);
}



struct enved_fft {
  mus_long_t size;
  mus_float_t *data;
  mus_float_t scale;
};


enved_fft *free_enved_fft(enved_fft *ef)
{
  if (ef)
    {
      if (ef->data) free(ef->data);
      ef->data = NULL;
      free(ef);
    }
  return(NULL);
}


void reflect_enved_fft_change(chan_info *cp)
{
  if ((enved_dialog_is_active()) &&
      (enved_target(ss) == ENVED_SPECTRUM) &&
      (cp == current_channel()))
    env_redisplay();
}



#define DEFAULT_ENVED_MAX_FFT_SIZE 1048576
static mus_long_t enved_max_fft_size = DEFAULT_ENVED_MAX_FFT_SIZE;


static enved_fft *make_enved_spectrum(chan_info *cp)
{
  enved_fft *ef;

  if (cp->edits[cp->edit_ctr]->fft == NULL)
    cp->edits[cp->edit_ctr]->fft = (enved_fft *)calloc(1, sizeof(enved_fft));
  ef = cp->edits[cp->edit_ctr]->fft;

  if ((ef) && 
      (ef->size == 0)) /* otherwise it is presumably already available */
    {
      mus_long_t i, data_len;
      mus_float_t data_max = 0.0;
      snd_fd *sf;

      data_len = cp->axis->hisamp - cp->axis->losamp;
      if (data_len > enved_max_fft_size)
	data_len = enved_max_fft_size;
      if (data_len == 0) return(NULL);

      sf = init_sample_read(cp->axis->losamp, cp, READ_FORWARD);
      if (sf == NULL) return(NULL);

      ef->size = snd_to_int_pow2(data_len);
      ef->data = (mus_float_t *)malloc(ef->size * sizeof(mus_float_t));
      if (ef->data == NULL) return(NULL);

      fourier_spectrum(sf, ef->data, ef->size, data_len, NULL, NULL);
      free_snd_fd(sf);
      for (i = 0; i < ef->size; i++) 
	if (ef->data[i] > data_max) 
	  data_max = ef->data[i];
      if (data_max > 0.0) ef->scale = data_max;
    }
  return(ef);
}


static void display_enved_spectrum(chan_info *cp, enved_fft *ef, axis_info *ap)
{
  if (ef)
    {
      mus_float_t incr, x = 0.0;
      int i = 0, j = 0;
      mus_long_t hisamp;
      mus_float_t samples_per_pixel, xf = 0.0, ina, ymax;
      ap->losamp = 0;
      ap->hisamp = ef->size - 1;
      ap->y0 = 0.0;
      ap->y1 = ef->scale;
      ap->x0 = 0.0;
      ap->x1 = SND_SRATE(cp->sound) / 2;
      init_axis_scales(ap);
      hisamp = ef->size / 2;
      incr = (mus_float_t)SND_SRATE(cp->sound) / (mus_float_t)(ef->size);
      samples_per_pixel = (mus_float_t)((double)hisamp / (mus_float_t)(ap->x_axis_x1 - ap->x_axis_x0));
      if ((samples_per_pixel < 4.0) &&
	  (hisamp < POINT_BUFFER_SIZE))
	{
	  for (i = 0, x = 0.0; i < hisamp; i++, x += incr)
	    set_grf_point(grf_x(x, ap), i, grf_y(ef->data[i], ap));
	  draw_grf_points(1, ap->ax, i, ap, 0.0, GRAPH_LINES);
	}
      else
	{
	  ymax = -1.0;
	  while (i < hisamp)
	    {
	      ina = ef->data[i++];
	      if (ina > ymax) ymax = ina;
	      xf += 1.0;
	      if (xf > samples_per_pixel)
		{
		  set_grf_point(grf_x(x, ap), j++, grf_y(ymax, ap));
		  x += (incr * samples_per_pixel); 
		  xf -= samples_per_pixel;
		  ymax = -1.0;
		}
	    }
	  draw_grf_points(1, ap->ax, j, ap, 0.0, GRAPH_LINES);
	}
    }
}


void enved_show_background_waveform(axis_info *ap, axis_info *gray_ap, bool apply_to_selection, bool show_fft, printing_t printing)
{
  int srate, pts = 0;
  graph_type_t old_time_graph_type = GRAPH_ONCE;
  mus_long_t samps;
  printing_t old_printing;
  bool two_sided = false;
  axis_info *active_ap = NULL;
  chan_info *active_channel = NULL;

  if (!(any_selected_sound())) return;

  if ((!gray_ap) || (!ap)) return;
  gray_ap->x_axis_x0 = ap->x_axis_x0;
  gray_ap->x_axis_x1 = ap->x_axis_x1;
  gray_ap->y_axis_y0 = ap->y_axis_y0;
  gray_ap->y_axis_y1 = ap->y_axis_y1;

  active_channel = current_channel();
  if ((!active_channel) || 
      (active_channel->active < CHANNEL_HAS_AXES) ||
      (active_channel->edits == NULL)) 
    return;

  old_printing = active_channel->printing;
  active_channel->printing = printing;

  if (show_fft)
    {
      if (enved_max_fft_size < transform_size(ss)) enved_max_fft_size = transform_size(ss);
      if (enved_max_fft_size < active_channel->transform_size) enved_max_fft_size = active_channel->transform_size;
      
#if USE_MOTIF
      if ((active_channel->transform_graph_type == GRAPH_AS_SONOGRAM) &&
	  (active_channel->graph_transform_p))
	{
	  /* if the sonogram isn't ready, try to get it 
	   *   this is for frequency envelopes as in animals.scm
	   */
	  if ((!(active_channel->fft_pix)) ||
	      (!(active_channel->fft_pix_ready)))
	    display_channel_fft_data(active_channel);
	}

      if ((active_channel->fft_pix) &&
	  (active_channel->fft_pix_ready) &&
	  (active_channel->transform_graph_type == GRAPH_AS_SONOGRAM) &&
	  (active_channel->graph_transform_p))
	{
	  int old_x0, old_y0;

	  old_x0 = active_channel->fft_pix_x0;
	  old_y0 = active_channel->fft_pix_y0; /* this actually aligns with the top of the enved window */
	  active_channel->fft_pix_x0 = ap->x_axis_x0;
	  active_channel->fft_pix_y0 = ap->y_axis_y1;

	  restore_fft_pix(active_channel, gray_ap->ax);

	  active_channel->fft_pix_x0 = old_x0;
	  active_channel->fft_pix_y0 = old_y0;
	}
      else display_enved_spectrum(active_channel, make_enved_spectrum(active_channel), gray_ap);
#else
      display_enved_spectrum(active_channel, make_enved_spectrum(active_channel), gray_ap);
#endif
    }
  else
    {
      active_ap = active_channel->axis;
      if (apply_to_selection)
	{
	  if (!(selection_is_active())) return;
	  samps = selection_len();
	  srate = selection_srate();
	  gray_ap->losamp = selection_beg(NULL);
	  gray_ap->hisamp = gray_ap->losamp + samps - 1;
	  gray_ap->x0 = (double)(gray_ap->losamp) / (double)srate;
	  gray_ap->x1 = (double)(gray_ap->hisamp) / (double)srate;
	  gray_ap->y0 = active_ap->y0;
	  gray_ap->y1 = active_ap->y1;
	}
      else
	{
	  /* show current channel overall view in gray scale */
	  samps = CURRENT_SAMPLES(active_channel);
	  srate = SND_SRATE(active_channel->sound);
	  gray_ap->losamp = 0;
	  gray_ap->hisamp = samps - 1;
	  if (active_channel->time_graph_type == GRAPH_AS_WAVOGRAM)
	    {
	      gray_ap->y0 = -1.0;
	      gray_ap->y1 = 1.0;
	    }
	  else
	    {
	      gray_ap->y0 = active_ap->y0;
	      gray_ap->y1 = active_ap->y1;
	    }
	  gray_ap->x0 = 0.0;
	  gray_ap->x1 = (double)samps / (double)srate;
	}
      init_axis_scales(gray_ap);
      active_channel->axis = gray_ap;
      old_time_graph_type = active_channel->time_graph_type;
      active_channel->time_graph_type = GRAPH_ONCE;
      pts = make_background_graph(active_channel, srate, &two_sided);
      active_channel->time_graph_type = old_time_graph_type;
      active_channel->axis = active_ap;
      if (pts > 0) 
	{
	  if (two_sided)
	    draw_both_grf_points(1, gray_ap->ax, pts, GRAPH_LINES);
	  else draw_grf_points(1, gray_ap->ax, pts, gray_ap, 0.0, GRAPH_LINES);
	}
    }
  active_channel->printing = old_printing;
}


env *enved_next_env(void)
{
  if (env_list_top > 0) 
    return(copy_env(env_list[env_list_top - 1])); 
  else return(NULL);
}


char *env_name_completer(widget_t w, const char *text, void *data)
{
  int matches = 0;
  char *current_match = NULL;
  if ((all_envs) && (text) && (*text))
    {
      int i, j, len, curlen;
      len = strlen(text);
      for (i = 0; i < all_envs_top; i++)
	if (strncmp(text, all_names[i], len) == 0)
	  {
	    matches++;
	    add_possible_completion(all_names[i]);
	    if (current_match == NULL)
	      current_match = mus_strdup(all_names[i]);
	    else 
	      {
		curlen = strlen(current_match);
		for (j = 0; j < curlen; j++)
		  if (current_match[j] != all_names[i][j])
		    {
		      current_match[j] = '\0';
		      break;
		    }
	      }
	  }
    }
  set_completion_matches(matches);
  if ((current_match) && (*current_match))
    return(current_match);
  return(mus_strdup(text));
}


void save_envelope_editor_state(FILE *fd)
{
  int i;
  for (i = 0; i < all_envs_top; i++)
    {
      char *estr;
      estr = env_to_string(all_envs[i]);
      if (estr)
	{
#if HAVE_SCHEME
	  fprintf(fd, "(%s %s %s %.4f)\n", S_define_envelope, all_names[i], estr, all_envs[i]->base);
#endif
#if HAVE_RUBY
	  {
	    char *name;
	    name = xen_scheme_procedure_to_ruby(S_define_envelope);
	    fprintf(fd, "%s(\"%s\", %s, %.4f)\n", name, all_names[i], estr, all_envs[i]->base);
	    free(name);
	  }
#endif
#if HAVE_FORTH
	  fprintf(fd, "\"%s\" %s %.4f %s drop\n", all_names[i], estr, all_envs[i]->base, S_define_envelope);
#endif
	  free(estr);
	}
    }
}


env *xen_to_env(XEN res)
{
  env *rtn = NULL;
  if (XEN_LIST_P(res))
    {
      int len = 0;
      len = XEN_LIST_LENGTH(res);
      if (len > 0)
	{
	  int i;
	  mus_float_t *data = NULL;
	  XEN lst;
	  
	  if (XEN_NUMBER_P(XEN_CAR(res)))
	    {
	      data = (mus_float_t *)calloc(len, sizeof(mus_float_t));
	      for (i = 0, lst = XEN_COPY_ARG(res); i < len; i++, lst = XEN_CDR(lst))
		{
		  XEN el;
		  el = XEN_CAR(lst);
		  data[i] = XEN_TO_C_DOUBLE_OR_ELSE(el, 0.0);
		}
	    }
	  else
	    {
	      /* embedded lists '((0 0) (100 1)) */
	      if (XEN_LIST_P(XEN_CAR(res)))
		{
		  len *= 2;
		  data = (mus_float_t *)calloc(len, sizeof(mus_float_t));
		  for (i = 0, lst = XEN_COPY_ARG(res); i < len; i += 2, lst = XEN_CDR(lst))
		    {
		      XEN el;
		      el = XEN_CAR(lst);
		      if ((!(XEN_NUMBER_P(XEN_CAR(el)))) ||
			  (!(XEN_NUMBER_P(XEN_CADR(el)))))
			{
			  free(data);
			  return(NULL);
			}
		      data[i] = XEN_TO_C_DOUBLE(XEN_CAR(el));
		      data[i + 1] = XEN_TO_C_DOUBLE(XEN_CADR(el));
		    }
		}
	      else
		{
		  /* something is screwed up */
		  return(NULL);
		}
	    }
	  if (data)
	    {
	      rtn = make_envelope(data, len);
	      free(data);
	    }
	}
    }
  return(rtn);
}


static bool x_increases(XEN res)
{
  int i, len;
  XEN lst;
  mus_float_t x;
  len = XEN_LIST_LENGTH(res);
  x = XEN_TO_C_DOUBLE(XEN_CAR(res));
  for (i = 2, lst = XEN_CDDR(XEN_COPY_ARG(res)); i < len; i += 2, lst = XEN_CDDR(lst))
    {
      mus_float_t nx;
      nx = XEN_TO_C_DOUBLE(XEN_CAR(lst));
      if (x >= nx) return(false);
      x = nx;
    }
  return(true);
}


#if (!HAVE_EXTENSION_LANGUAGE)
  #define ENV_BUFFER_SIZE 128
  static int env_buffer_size = 0;
  static mus_float_t *env_buffer = NULL;
  static char env_white_space[5] = {' ', '(', ')', '\t', '\''};
#endif


env *string_to_env(const char *str) 
{
#if HAVE_EXTENSION_LANGUAGE
  XEN res;
  int len = 0;
  res = snd_catch_any(eval_str_wrapper, (void *)str, "string->env");
  if (XEN_LIST_P_WITH_LENGTH(res, len))
    {
      if ((len % 2) == 0)
	{
	  if (x_increases(res))
	    return(xen_to_env(res));
	  else snd_error("x axis points not increasing: %s", str);
	}
      else snd_error("odd length envelope? %s", str);
    }
  else snd_error("%s is not a list", str);
  return(NULL);
#else
  char *tok, *tmp;
  int i;
  float f;
  if ((str) && (*str))
    {
      char *old_tmp;
      tmp = mus_strdup(str);
      old_tmp = tmp;
      i = 0;
      if (env_buffer_size == 0)
	{
	  env_buffer_size = ENV_BUFFER_SIZE;
	  env_buffer = (mus_float_t *)calloc(ENV_BUFFER_SIZE, sizeof(mus_float_t));
	}
      if ((*tmp) == '\'') tmp++;
      if ((*tmp) == '(') tmp++;
      tok = strtok(tmp, env_white_space);
      while (tok)
	{
	  if (!(sscanf(tok, "%f", &f)))
	    {
	      snd_error("%s in env list is not a number", tok);
	      return(NULL);
	    }
	  env_buffer[i] = (mus_float_t)f;
	  i++;
	  if (i == env_buffer_size)
	    {
	      env_buffer_size *= 2;
	      env_buffer = (mus_float_t *)realloc(env_buffer, env_buffer_size * sizeof(mus_float_t));
	    }
	  tok = strtok(NULL, env_white_space);
	}
      if ((i == 0) || (i & 1)) 
	snd_error("odd length envelope? %s", str);
      free(old_tmp);
      return(make_envelope(env_buffer, i));
    }
  return(NULL);
#endif
}


env *position_to_env(int pos)
{
  if (pos < 0) return(NULL);
  return(copy_env(all_envs[pos]));
}


env *name_to_env(const char *str)
{
  env *e;
  int pos;
  if ((!str) || (!(*str))) return(NULL);
  pos = find_env(str);
  if (pos >= 0) return(copy_env(all_envs[pos]));
#if HAVE_SCHEME || HAVE_FORTH
  e = xen_to_env(XEN_NAME_AS_C_STRING_TO_VALUE(str));
#else
  e = xen_to_env(XEN_EVAL_C_STRING((char *)str));
#endif
  return(e);
}


#if HAVE_RUBY
#define SND_ENV_MAX_VARS 100
static XEN snd_env_array[SND_ENV_MAX_VARS];
static int env_index = -1;
#endif


static XEN g_define_envelope(XEN name, XEN data, XEN base)
{
  env *e;
  const char *ename;

  #define H_define_envelope "(" S_define_envelope " name data :optional base): load 'name' with associated 'data', a list of breakpoints \
into the envelope editor."

  XEN_ASSERT_TYPE(XEN_STRING_P(name) || XEN_SYMBOL_P(name), name, XEN_ARG_1, S_define_envelope, "a string or symbol");
  XEN_ASSERT_TYPE(XEN_LIST_P(data), data, XEN_ARG_2, S_define_envelope, "a list of breakpoints");
  XEN_ASSERT_TYPE(XEN_NUMBER_IF_BOUND_P(base) || XEN_FALSE_P(base), base, XEN_ARG_3, S_define_envelope, "a float or " PROC_FALSE);

  if (XEN_STRING_P(name))
    ename = XEN_TO_C_STRING(name);
  else ename = XEN_SYMBOL_TO_C_STRING(name);

  e = xen_to_env(data);
  if (!e) return(XEN_FALSE);

  if (XEN_NUMBER_P(base))
    e->base = XEN_TO_C_DOUBLE(base);

#if HAVE_RUBY
  {
    char *name;
    alert_envelope_editor(name = xen_scheme_global_variable_to_ruby(ename), e);
    if (env_index >= SND_ENV_MAX_VARS)
      env_index = 0;
    else
      env_index++;
    XEN_DEFINE_VARIABLE(ename, snd_env_array[env_index], data); /* need global C variable */
    free(name);
    return(snd_env_array[env_index]);
  }
#endif

#if HAVE_SCHEME || HAVE_FORTH
  {
    XEN temp;
    alert_envelope_editor(ename, e);
    XEN_DEFINE_VARIABLE(ename, temp, data); /* already gc protected */
    return(temp);
  }
#endif
}


XEN env_to_xen(env *e)
{
  if (e) 
    return(mus_array_to_list(e->data, 0, e->pts * 2));
  return(XEN_EMPTY_LIST);
}


void add_or_edit_symbol(const char *name, env *val)
{
  /* called from envelope editor -- pass new definition into scheme */
#if HAVE_RUBY
  char *buf, *tmpstr = NULL;
  int len;

  if (!val) return;
  tmpstr = env_to_string(val);
  len = mus_strlen(tmpstr) + mus_strlen(name) + 32;
  buf = (char *)calloc(len, sizeof(char));
  mus_snprintf(buf, len, "%s = %s", name, tmpstr);
  if (tmpstr) free(tmpstr);

  snd_catch_any(eval_str_wrapper, buf, buf);

  free(buf);
#endif

#if HAVE_SCHEME
  XEN e;
  if (!val) return;
  if (XEN_DEFINED_P(name))
    {
      e = XEN_NAME_AS_C_STRING_TO_VARIABLE(name);
      XEN_VARIABLE_SET(e, env_to_xen(val));
    }
  else XEN_DEFINE_VARIABLE(name, e, env_to_xen(val));
#endif

#if HAVE_FORTH
  XEN e;
  if (!val) return;
  if (XEN_DEFINED_P(name))
    e = XEN_VARIABLE_SET(name, env_to_xen(val));
  else XEN_DEFINE_VARIABLE(name, e, env_to_xen(val));
#endif
}


env *get_env(XEN e, const char *origin) /* list in e */
{
  int i, len = 0;
  env *new_env;

  XEN_ASSERT_TYPE(XEN_LIST_P_WITH_LENGTH(e, len), e, XEN_ARG_1, origin, "a list");

  if (len == 0)
    XEN_ERROR(NO_DATA,
	      XEN_LIST_3(C_TO_XEN_STRING("~A: null env, ~A"), 
			 C_TO_XEN_STRING(origin), 
			 e));
  new_env = xen_to_env(e);
  if (!new_env)
    XEN_ERROR(XEN_ERROR_TYPE("env-error"),
	      XEN_LIST_2(C_TO_XEN_STRING("envelope break point list is screwed up: ~A"),
			 e));

  for (i = 2; i < new_env->pts * 2; i += 2)
    if (new_env->data[i - 2] > new_env->data[i])
      {
	XEN msg;
	char *buf;
	buf = (char *)calloc(1024, sizeof(char));
	mus_snprintf(buf, 1024, "%s: env at breakpoint %d: x axis value %f > %f", origin, i / 2, new_env->data[i - 2], new_env->data[i]);
	msg = C_TO_XEN_STRING(buf);
	free(buf);
	free_env(new_env);
	XEN_ERROR(XEN_ERROR_TYPE("env-error"),
		  XEN_LIST_3(C_TO_XEN_STRING("~A, ~A"),
			     msg,
			     e));
      }
  return(new_env);
}


static XEN g_save_envelopes(XEN filename)
{
  #define H_save_envelopes "(" S_save_envelopes " :optional filename): save the envelopes known to the envelope editor in filename"
  char *name = NULL;
  FILE *fd;

  XEN_ASSERT_TYPE((XEN_STRING_P(filename) || (XEN_FALSE_P(filename)) || (XEN_NOT_BOUND_P(filename))), 
		  filename, XEN_ONLY_ARG, S_save_envelopes, "a string or " PROC_FALSE);
  if (XEN_STRING_P(filename)) 
    name = mus_expand_filename(XEN_TO_C_STRING(filename));
  else name = mus_strdup("envs.save");
  
  if (name)
    {
      fd = FOPEN(name, "w");
      if (fd) 
	{
	  save_envelope_editor_state(fd);
	  snd_fclose(fd, name);
	}
      free(name);

      if (!fd)
	{
	  XEN_ERROR(CANNOT_SAVE,
		    XEN_LIST_3(C_TO_XEN_STRING(S_save_envelopes ": can't save ~S, ~A"),
			       filename,
			       C_TO_XEN_STRING(snd_open_strerror())));
	}
    }
  return(filename);
}


static XEN enved_hook;

static bool check_enved_hook(env *e, int pos, mus_float_t x, mus_float_t y, enved_point_t reason)
{
  bool env_changed = false;
  if (XEN_HOOKED(enved_hook))
    {
      int len = 0;
#if HAVE_SCHEME
      int gc_loc;
#endif
      XEN result = XEN_FALSE;
      XEN procs, env_list;
      /* if hook procedure returns a list, that is the new contents of the
       * envelope -- if its length doesn't match current, we need to remake
       * current. Otherwise return 0, and assume the caller will handle default
       */
      procs = XEN_HOOK_PROCEDURES(enved_hook);
      env_list = env_to_xen(e);
#if HAVE_SCHEME
      gc_loc = s7_gc_protect(s7, env_list);
#endif

      while (XEN_NOT_NULL_P(procs))
	{
	  result = XEN_APPLY(XEN_CAR(procs), 
			     XEN_LIST_5(env_list,
					C_TO_XEN_INT(pos),
					C_TO_XEN_DOUBLE(x),
					C_TO_XEN_DOUBLE(y),
					C_TO_XEN_INT((int)reason)),
			     S_enved_hook);
	  procs = XEN_CDR (procs);
	  if ((XEN_NOT_FALSE_P(result)) && 
	      (XEN_LIST_P_WITH_LENGTH(result, len)))
	    {
	      /* remake env and (if not null procs) env_list */
	      /* each successive hook procedure gets the on-going (changing) envelope */
	      int i;
	      XEN lst;
	      if (len > e->data_size)
		{
		  free(e->data);
		  e->data = (mus_float_t *)calloc(len, sizeof(mus_float_t));
		  e->data_size = len;
		}
	      e->pts = len / 2;
	      for (i = 0, lst = XEN_COPY_ARG(result); i < len; i++, lst = XEN_CDR(lst))
		e->data[i] = XEN_TO_C_DOUBLE(XEN_CAR(lst));
	      if (XEN_NOT_NULL_P(procs))
		{
#if HAVE_SCHEME
		  s7_gc_unprotect_at(s7, gc_loc);
#endif
		  env_list = env_to_xen(e);
#if HAVE_SCHEME
		  gc_loc = s7_gc_protect(s7, env_list);
#endif
	
		}
	      env_changed = true;
	    }
	}
#if HAVE_SCHEME
      s7_gc_unprotect_at(s7, gc_loc);
#endif
    }
  return(env_changed); /* 0 = default action */
}


static XEN g_enved_base(void) {return(C_TO_XEN_DOUBLE(enved_base(ss)));}

static XEN g_set_enved_base(XEN val) 
{
  #define H_enved_base "(" S_enved_base "): envelope editor exponential base value (1.0)"
  XEN_ASSERT_TYPE(XEN_NUMBER_P(val), val, XEN_ONLY_ARG, S_setB S_enved_base, "a number"); 
  set_enved_base(mus_fclamp(0.0, XEN_TO_C_DOUBLE(val), 300000.0));
  return(C_TO_XEN_DOUBLE(enved_base(ss)));
}


static XEN g_enved_power(void) {return(C_TO_XEN_DOUBLE(enved_power(ss)));}

static XEN g_set_enved_power(XEN val) 
{
  #define H_enved_power "(" S_enved_power "): envelope editor base scale range (9.0^power)"
  XEN_ASSERT_TYPE(XEN_NUMBER_P(val), val, XEN_ONLY_ARG, S_setB S_enved_power, "a number"); 
  set_enved_power(mus_fclamp(0.0, XEN_TO_C_DOUBLE(val), 10.0));
  return(C_TO_XEN_DOUBLE(enved_power(ss)));
}


static XEN g_enved_clip_p(void) {return(C_TO_XEN_BOOLEAN(enved_clip_p(ss)));}

static XEN g_set_enved_clip_p(XEN on)
{
  #define H_enved_clip_p "(" S_enved_clip_p "): envelope editor clip button setting; \
if clipping, the motion of the mouse is restricted to the current graph bounds."

  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(on), on, XEN_ONLY_ARG, S_setB S_enved_clip_p, "a boolean");
  set_enved_clip_p(XEN_TO_C_BOOLEAN(on)); 
  return(C_TO_XEN_BOOLEAN(enved_clip_p(ss)));
}


static XEN g_enved_style(void) {return(C_TO_XEN_INT(enved_style(ss)));}

static XEN g_set_enved_style(XEN val) 
{
  #define H_enved_style "(" S_enved_style "): envelope editor breakpoint connection choice: can \
be " S_envelope_linear ", or " S_envelope_exponential

  int choice;
  XEN_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ONLY_ARG, S_setB S_enved_style, S_envelope_linear ", or " S_envelope_exponential);
  choice = XEN_TO_C_INT(val);
  if ((choice == ENVELOPE_LINEAR) || (choice == ENVELOPE_EXPONENTIAL))
    {
      set_enved_style((env_type_t)choice);
      reflect_enved_style();
    }
  else XEN_OUT_OF_RANGE_ERROR(S_enved_style, XEN_ONLY_ARG, val, "must be " S_envelope_linear ", or " S_envelope_exponential);
  return(val);
}


static XEN g_enved_target(void) {return(C_TO_XEN_INT((int)enved_target(ss)));}

static XEN g_set_enved_target(XEN val) 
{
  enved_target_t n;
  int in_n;

  #define H_enved_target "(" S_enved_target "): determines how the envelope edit envelope is applied; \
choices are " S_enved_amplitude ", " S_enved_srate "(apply to speed), and " S_enved_spectrum "(apply as a filter)."

  XEN_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ONLY_ARG, S_setB S_enved_target, "an integer"); 
  in_n = XEN_TO_C_INT(val);
  if (in_n < 0)
    XEN_OUT_OF_RANGE_ERROR(S_setB S_enved_target, 1, val, "~A, but must be " S_enved_amplitude ", " S_enved_srate ", or " S_enved_spectrum);
  if (in_n > (int)ENVED_SRATE)
    XEN_OUT_OF_RANGE_ERROR(S_setB S_enved_target, 1, val, "~A, but must be " S_enved_amplitude ", " S_enved_srate ", or " S_enved_spectrum);
  n = (enved_target_t)in_n;
  set_enved_target(n); 
  return(C_TO_XEN_INT((int)enved_target(ss)));
}


static XEN g_enved_wave_p(void) {return(C_TO_XEN_BOOLEAN(enved_wave_p(ss)));}

static XEN g_set_enved_wave_p(XEN val) 
{
  #define H_enved_wave_p "(" S_enved_wave_p "): " PROC_TRUE " if the envelope editor is displaying the waveform to be edited"
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(val), val, XEN_ONLY_ARG, S_setB S_enved_wave_p, "a boolean");
  set_enved_wave_p(XEN_TO_C_BOOLEAN(val));
  return(C_TO_XEN_BOOLEAN(enved_wave_p(ss)));
}


static XEN g_enved_in_dB(void) {return(C_TO_XEN_BOOLEAN(enved_in_dB(ss)));}

static XEN g_set_enved_in_dB(XEN val) 
{
  #define H_enved_in_dB "(" S_enved_in_dB "): " PROC_TRUE " if the envelope editor is using dB"
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(val), val, XEN_ONLY_ARG, S_setB S_enved_in_dB, "a boolean");
  set_enved_in_dB(XEN_TO_C_BOOLEAN(val)); 
  return(C_TO_XEN_BOOLEAN(enved_in_dB(ss)));
}


static XEN g_enved_filter_order(void) {return(C_TO_XEN_INT(enved_filter_order(ss)));}

static XEN g_set_enved_filter_order(XEN val) 
{
  #define H_enved_filter_order "(" S_enved_filter_order "): envelope editor's FIR filter order (40)"
  XEN_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ONLY_ARG, S_setB S_enved_filter_order, "an integer"); 
  set_enved_filter_order(XEN_TO_C_INT(val));
  return(C_TO_XEN_INT(enved_filter_order(ss)));
}


static XEN g_enved_dialog(void) 
{
  #define H_enved_dialog "(" S_enved_dialog "): start the Envelope Editor"
  return(XEN_WRAP_WIDGET(create_envelope_editor()));
}


#ifdef XEN_ARGIFY_1
XEN_NARGIFY_0(g_enved_base_w, g_enved_base)
XEN_NARGIFY_1(g_set_enved_base_w, g_set_enved_base)
XEN_NARGIFY_0(g_enved_power_w, g_enved_power)
XEN_NARGIFY_1(g_set_enved_power_w, g_set_enved_power)
XEN_NARGIFY_0(g_enved_clip_p_w, g_enved_clip_p)
XEN_NARGIFY_1(g_set_enved_clip_p_w, g_set_enved_clip_p)
XEN_NARGIFY_0(g_enved_style_w, g_enved_style)
XEN_NARGIFY_1(g_set_enved_style_w, g_set_enved_style)
XEN_NARGIFY_0(g_enved_target_w, g_enved_target)
XEN_NARGIFY_1(g_set_enved_target_w, g_set_enved_target)
XEN_NARGIFY_0(g_enved_wave_p_w, g_enved_wave_p)
XEN_NARGIFY_1(g_set_enved_wave_p_w, g_set_enved_wave_p)
XEN_NARGIFY_0(g_enved_in_dB_w, g_enved_in_dB)
XEN_NARGIFY_1(g_set_enved_in_dB_w, g_set_enved_in_dB)
XEN_NARGIFY_0(g_enved_filter_order_w, g_enved_filter_order)
XEN_NARGIFY_1(g_set_enved_filter_order_w, g_set_enved_filter_order)
XEN_NARGIFY_0(g_enved_dialog_w, g_enved_dialog)
XEN_ARGIFY_1(g_save_envelopes_w, g_save_envelopes)
XEN_ARGIFY_3(g_define_envelope_w, g_define_envelope)
#else
#define g_enved_base_w g_enved_base
#define g_set_enved_base_w g_set_enved_base
#define g_enved_power_w g_enved_power
#define g_set_enved_power_w g_set_enved_power
#define g_enved_clip_p_w g_enved_clip_p
#define g_set_enved_clip_p_w g_set_enved_clip_p
#define g_enved_style_w g_enved_style
#define g_set_enved_style_w g_set_enved_style
#define g_enved_target_w g_enved_target
#define g_set_enved_target_w g_set_enved_target
#define g_enved_wave_p_w g_enved_wave_p
#define g_set_enved_wave_p_w g_set_enved_wave_p
#define g_enved_in_dB_w g_enved_in_dB
#define g_set_enved_in_dB_w g_set_enved_in_dB
#define g_enved_filter_order_w g_enved_filter_order
#define g_set_enved_filter_order_w g_set_enved_filter_order
#define g_enved_dialog_w g_enved_dialog
#define g_save_envelopes_w g_save_envelopes
#define g_define_envelope_w g_define_envelope
#endif

void g_init_env(void)
{
  #define H_enved_amplitude "The value for " S_enved_target " that sets the envelope editor 'amp' button."
  #define H_enved_spectrum "The value for " S_enved_target " that sets the envelope editor 'flt' button."
  #define H_enved_srate "The value for " S_enved_target " that sets the envelope editor 'src' button."

  XEN_DEFINE_CONSTANT(S_enved_amplitude, ENVED_AMPLITUDE, H_enved_amplitude);
  XEN_DEFINE_CONSTANT(S_enved_spectrum,  ENVED_SPECTRUM,  H_enved_spectrum);
  XEN_DEFINE_CONSTANT(S_enved_srate,     ENVED_SRATE,     H_enved_srate);

  XEN_DEFINE_CONSTANT(S_envelope_linear,      ENVELOPE_LINEAR,      S_enved_style " choice: linear connections between breakpoints");
  XEN_DEFINE_CONSTANT(S_envelope_exponential, ENVELOPE_EXPONENTIAL, S_enved_style " choice: exponential connections between breakpoints");

  XEN_DEFINE_PROCEDURE_WITH_SETTER(S_enved_base,   g_enved_base_w,   H_enved_base,   S_setB S_enved_base,   g_set_enved_base_w,    0, 0, 1, 0);
  XEN_DEFINE_PROCEDURE_WITH_SETTER(S_enved_power,  g_enved_power_w,  H_enved_power,  S_setB S_enved_power,  g_set_enved_power_w,   0, 0, 1, 0);
  XEN_DEFINE_PROCEDURE_WITH_SETTER(S_enved_clip_p, g_enved_clip_p_w, H_enved_clip_p, S_setB S_enved_clip_p, g_set_enved_clip_p_w,  0, 0, 1, 0);
  XEN_DEFINE_PROCEDURE_WITH_SETTER(S_enved_style,  g_enved_style_w,  H_enved_style,  S_setB S_enved_style,  g_set_enved_style_w,   0, 0, 1, 0);
  XEN_DEFINE_PROCEDURE_WITH_SETTER(S_enved_target, g_enved_target_w, H_enved_target, S_setB S_enved_target, g_set_enved_target_w,  0, 0, 1, 0);
  XEN_DEFINE_PROCEDURE_WITH_SETTER(S_enved_wave_p, g_enved_wave_p_w, H_enved_wave_p, S_setB S_enved_wave_p, g_set_enved_wave_p_w,  0, 0, 1, 0);
  XEN_DEFINE_PROCEDURE_WITH_SETTER(S_enved_in_dB,  g_enved_in_dB_w,  H_enved_in_dB,  S_setB S_enved_in_dB,  g_set_enved_in_dB_w,   0, 0, 1, 0);

  XEN_DEFINE_PROCEDURE_WITH_SETTER(S_enved_filter_order, g_enved_filter_order_w, H_enved_filter_order,
				   S_setB S_enved_filter_order, g_set_enved_filter_order_w,  0, 0, 1, 0);

  XEN_DEFINE_PROCEDURE(S_enved_dialog,    g_enved_dialog_w,    0, 0, 0, H_enved_dialog);
  XEN_DEFINE_PROCEDURE(S_save_envelopes,  g_save_envelopes_w,  0, 1, 0, H_save_envelopes);


#if HAVE_SCHEME
  XEN_DEFINE_PROCEDURE(S_define_envelope "-1", g_define_envelope_w, 2, 1, 0, H_define_envelope);
  XEN_EVAL_C_STRING("(defmacro define-envelope (a . b) `(define-envelope-1 ',a ,@b))");
  XEN_EVAL_C_STRING("(defmacro defvar (a b) `(define-envelope-1 ',a ,b))");
  /* macro used here to ensure that "ampf" in (defvar ampf '(0 0 1 1)) is not evaluated */
#else
  XEN_DEFINE_PROCEDURE(S_define_envelope, g_define_envelope_w, 2, 1, 0, H_define_envelope);
#endif

  XEN_DEFINE_CONSTANT(S_enved_add_point,      ENVED_ADD_POINT,      S_enved_hook " 'reason' arg when point is added");
  XEN_DEFINE_CONSTANT(S_enved_delete_point,   ENVED_DELETE_POINT,   S_enved_hook " 'reason' arg when point is deleted");
  XEN_DEFINE_CONSTANT(S_enved_move_point,     ENVED_MOVE_POINT,     S_enved_hook " 'reason' arg when point is moved");

#if HAVE_SCHEME
  #define H_enved_hook S_enved_hook " (env pt new-x new-y reason): \
called each time a breakpoint is changed in the envelope editor; \
if it returns a list, that list defines the new envelope, \
otherwise the breakpoint is moved (but not beyond the neighboring \
breakpoint), leaving other points untouched.  The kind of change that triggered the hook \
is 'reason' which can be " S_enved_move_point ", " S_enved_delete_point ", \
or " S_enved_add_point ".  This hook makes it possible to define attack \
and decay portions in the envelope editor, or use functions such as \
stretch-envelope from env.scm: \n\
 (add-hook! " S_enved_hook "\n\
   (lambda (env pt x y reason)\n\
     (if (= reason " S_enved_move_point ")\n\
         (let* ((old-x (list-ref env (* pt 2)))\n\
                (new-env (stretch-envelope env old-x x)))\n\
           (list-set! new-env (+ (* pt 2) 1) y)\n\
           new-env)\n\
         #f)))"
#endif
#if HAVE_RUBY
  #define H_enved_hook S_enved_hook " (env pt new-x new-y reason): \
called each time a breakpoint is changed in the envelope editor; \
if it returns a list, that list defines the new envelope, \
otherwise the breakpoint is moved (but not beyond the neighboring \
breakpoint), leaving other points untouched.  The kind of change that triggered the hook \
is 'reason' which can be " S_enved_move_point ", " S_enved_delete_point ", \
or " S_enved_add_point ".  This hook makes it possible to define attack \
and decay portions in the envelope editor."
#endif
#if HAVE_FORTH
  #define H_enved_hook S_enved_hook " (env pt new-x new-y reason): \
called each time a breakpoint is changed in the envelope editor; \
if it returns a list, that list defines the new envelope, \
otherwise the breakpoint is moved (but not beyond the neighboring \
breakpoint), leaving other points untouched.  The kind of change that triggered the hook \
is 'reason' which can be " S_enved_move_point ", " S_enved_delete_point ", \
or " S_enved_add_point ".  This hook makes it possible to define attack \
and decay portions in the envelope editor, or use functions such as \
stretch-envelope from env.fth: \n\
" S_enved_hook " lambda: <{ en pt x y reason }>\n\
  reason " S_enved_move_point " = if\n\
    en old-x  en pt 2* list@  x stretch-envelope  pt 2* 1+ y list!\n\
  else\n\
    #f\n\
  then\n\
; add-hook!"
#endif

  enved_hook = XEN_DEFINE_HOOK(S_enved_hook, 5, H_enved_hook);

  ss->enved = new_env_editor();
  free(ss->enved->axis);
  ss->enved->axis = NULL;
  ss->enved->in_dB = DEFAULT_ENVED_IN_DB;
  ss->enved->clip_p = DEFAULT_ENVED_CLIP_P;
}
