/*******************************************************************************
* FILE NAME: iddeinfo.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in iddeinfo.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
extern "C"
  {
  #define INCL_WINDDE
  #define INCL_WINATOM
  #define INCL_DOSMEMMGR
  #define INCL_DOSERRORS
  #include <os2.h>
  #include <string.h>
  }

#include <iddeinfo.hpp>
#include <iddeevt.hpp>
#include <itrace.hpp>
#include <iexcept.hpp>
#include <iqueue.h>

class IDDEAckQueue : public IQueue<IDDEServerAcknowledgeEvent*> {
public:
   IDDEAckQueue ( );
  ~IDDEAckQueue ( );
}; // IDDEAckQueue

/*------------------------------------------------------------------------------
| IDdeleteElemAQ                                                               |
------------------------------------------------------------------------------*/
static Boolean
  IDdeleteElemAQ ( IDDEServerAcknowledgeEvent* const& ackEvent,
                   void*                      anything )
{
   delete (IDDEServerAcknowledgeEvent*)ackEvent;
   return true;
}

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IDDEINFO_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| IDDEInfo__stringFromAtom                                                     |
|                                                                              |
| Query the system atom table for the string name of an atom.                  |
------------------------------------------------------------------------------*/
IString IDDEInfo__stringFromAtom ( unsigned long atom )
{
   IMODTRACE_DEVELOP("IDDEInfo::stringFromAtom");
   IASSERT(atom != 0);
   char szBuf[256];
   szBuf[0] = '\0';
   ITRACE_ALL(IString("Atom is ") + IString(atom));
   unsigned long rc = WinQueryAtomName(WinQuerySystemAtomTable(),
                                       (ATOM)atom,
                                       (PSZ)szBuf, 256);
   ITRACE_ALL(IString("Atom string is ") + IString(szBuf));
   if (!rc)
      ITHROWGUIERROR2("WinQueryAtomName",IErrorInfo::invalidParameter,
          IException::recoverable);
   return szBuf;
}

/*------------------------------------------------------------------------------
| IDDEInfo__atomFromString                                                     |
|                                                                              |
| Query the system atom table for the atom from a string name.                 |
------------------------------------------------------------------------------*/
unsigned long IDDEInfo__atomFromString ( const char* atomAsString,
                                         Boolean*    atomAdded )
{
   IMODTRACE_DEVELOP("IDDEInfo::atomFromString");
   ITRACE_ALL(IString("Atom string is ") + IString(atomAsString));
   IASSERT(atomAsString!=0);

   unsigned long ulAtom = WinFindAtom(WinQuerySystemAtomTable(),
                                      (PSZ)atomAsString);
   if (!ulAtom)
   {
      *atomAdded = true;
      ulAtom = WinAddAtom(WinQuerySystemAtomTable(),
                          (PSZ)atomAsString);
      ITRACE_ALL(IString("Atom is ") + IString(ulAtom));
      if (!ulAtom)
         ITHROWGUIERROR2("WinAddAtom",IErrorInfo::invalidParameter,
            IException::recoverable);
   }
   else
      *atomAdded = false;

   return ulAtom;
}

/*------------------------------------------------------------------------------
| IDDEInfo__freeMemory                                                         |
------------------------------------------------------------------------------*/
void IDDEInfo__freeMemory ( _DDESTRUCT* ddeInfo )
{
   IMODTRACE_DEVELOP("IDDEInfo ::freeMemory");
   unsigned long rc = 0;
   unsigned long j = 0;
   do {
   rc = DosFreeMem((PVOID)ddeInfo);
   } while ((rc == ERROR_INTERRUPT) && (++j <= 3));
   if (rc)
      ITHROWSYSTEMERROR(rc,"DosFreeMem",IErrorInfo::accessError,
        IException::recoverable);
}

/*------------------------------------------------------------------------------
| IDDEInfo__buildDDEStruct                                                     |
|                                                                              |
| Build a DDE structure in shared memory, making the object giveable.          |
------------------------------------------------------------------------------*/
_DDESTRUCT* IDDEInfo__buildDDEStruct ( const char* itemName,
                                       const char* dataFormat,
                                       unsigned short status,
                                       const void* xferData,
                                       unsigned long dataLength,
                                       Boolean* atomAdded )
{
   IMODTRACE_DEVELOP("IDDEInfo::buildDDEStruct");
   ITRACE_ALL(IString("Item is ") + IString(itemName));
   ITRACE_ALL(IString("Format is ") + IString(dataFormat));
   ITRACE_ALL(IString("Status is ") + IString(status));
   ITRACE_ALL(IString("Data is ") + IString(xferData, dataLength));
   PDDESTRUCT pddes;
   void* pbuf;
   unsigned long rc = 0;
   unsigned long j = 0;
   unsigned long ulItemLen = 1;
   if (itemName != 0)
      ulItemLen = strlen(itemName) + 1;
//  Allocate the shared memory
   do {
   rc = DosAllocSharedMem(&pbuf,
                          0,
                          sizeof(DDESTRUCT) + ulItemLen + dataLength,
                          PAG_COMMIT | PAG_READ | PAG_WRITE | OBJ_GIVEABLE);
   } while ((rc == ERROR_INTERRUPT) && (++j <= 3));
   if (rc)
      ITHROWSYSTEMERROR(rc,"DosAllocSharedMem",IErrorInfo::outOfMemory,
         IException::recoverable);
// Fill in the new DDE structure
   pddes = (PDDESTRUCT)pbuf;
   pddes->cbData = dataLength;
   pddes->fsStatus = status;
   if (dataFormat != 0 && *dataFormat != '\0')
      pddes->usFormat = IDDEInfo__atomFromString(dataFormat, atomAdded);
   else
      pddes->usFormat = 0;
   pddes->offszItemName = sizeof(DDESTRUCT);
   pddes->offabData = sizeof(DDESTRUCT) + ulItemLen;
// copy the item name
   if (itemName !=0)
      strcpy(DDES_PSZITEMNAME(pddes), itemName);
   else
      strcpy(DDES_PSZITEMNAME(pddes), "");
// copy the data
   if (xferData !=0 && dataLength != 0)
      memcpy(DDES_PABDATA(pddes), xferData, dataLength);
   return pddes;
}

/*------------------------------------------------------------------------------
| IDDEFormat::IDDEFormat                                                       |
------------------------------------------------------------------------------*/
IDDEFormat :: IDDEFormat ( unsigned long ddeFormat )
    : ulClFormat(ddeFormat), pNextCl(0)
{
   ;
}

/*------------------------------------------------------------------------------
| IDDEFormat::~IDDEFormat                                                      |
|                                                                              |
| Destructor for IDDEFormat. Remove any formats added from the system atom     |
| table.                                                                       |
------------------------------------------------------------------------------*/
IDDEFormat :: ~IDDEFormat ( )
{
   if (pNextCl)
      delete pNextCl;
   ATOM atom = WinDeleteAtom(WinQuerySystemAtomTable(), ulClFormat);
}

/*------------------------------------------------------------------------------
| IDDEFormatSet::IDDEFormatSet                                                 |
------------------------------------------------------------------------------*/
IDDEFormatSet :: IDDEFormatSet ( )
      : pFirstCl(0)
{
   ;
}

/*------------------------------------------------------------------------------
| IDDEFormatSet::~IDDEFormatSet                                                |
------------------------------------------------------------------------------*/
IDDEFormatSet :: ~IDDEFormatSet ( )
{
   if (pFirstCl)
      delete pFirstCl;
}

/*------------------------------------------------------------------------------
| IDDEFormatSet::add                                                           |
------------------------------------------------------------------------------*/
void IDDEFormatSet :: add ( unsigned long ddeFormat )
{
   this->priResCl.lock();
   if (!pFirstCl)
      pFirstCl = new IDDEFormat(ddeFormat);
   else {
      IDDEFormat *pNext = pFirstCl,
                 *pCurrent;
      while ( pNext ) {
         pCurrent = pNext;
         pNext = pNext->pNextCl;
      }
      pCurrent->pNextCl = new IDDEFormat( ddeFormat );
   }
   this->priResCl.unlock();
}

/*------------------------------------------------------------------------------
| IDDEAckQueue::IDDEAckQueue                                                   |
------------------------------------------------------------------------------*/
IDDEAckQueue :: IDDEAckQueue ( )
      : IQueue<IDDEServerAcknowledgeEvent*>(3)
{
   ;
}

/*------------------------------------------------------------------------------
| IDDEAckQueue::~IDDEAckQueue                                                  |
------------------------------------------------------------------------------*/
IDDEAckQueue :: ~IDDEAckQueue ( )
{
   ;
}

/*------------------------------------------------------------------------------
| IDDEServerConversation::~IDDEServerConversation                              |
------------------------------------------------------------------------------*/
IDDEServerConversation :: ~IDDEServerConversation ( )
{
   if (pAckQCl)
   {
      pAckQCl->allElementsDo(&IDdeleteElemAQ);
      pAckQCl->removeAll();
      delete pAckQCl;
   }
}

/*------------------------------------------------------------------------------
| IDDEServerConversation::eventCount                                           |
------------------------------------------------------------------------------*/
unsigned long IDDEServerConversation :: eventCount ( ) const
{
   if (pAckQCl)
      return (pAckQCl->numberOfElements());
   return 0;
}

/*------------------------------------------------------------------------------
| IDDEServerConversation::isAckOutstanding                                     |
------------------------------------------------------------------------------*/
Boolean IDDEServerConversation :: isAckOutstanding ( IString item,
                                                     IString format ) const
{
   IMODTRACE_DEVELOP("IDDEServerConversation::isAckOutstanding");
   Boolean bFound = false;
   if (pAckQCl)
   {
      ITRACE_DEVELOP("Number of elements in Q is: " +
          IString(pAckQCl->numberOfElements()));
      IDDEServerAcknowledgeEvent* pEvt;
      IDDEAckQueue::Cursor myCurs(*pAckQCl);
      IString inItem = IString::upperCase(item);
      IString inFormat = IString::upperCase(format);
      IString qItem;
      IString qFormat;
      forCursor(myCurs)
      {
         pEvt = pAckQCl->elementAt(myCurs);
         qItem = IString::upperCase(pEvt->item());
         qFormat = IString::upperCase(pEvt->format());
         if (inItem == qItem && inFormat == qFormat)
            bFound = true;
      }
   }
   else
      ITRACE_DEVELOP("Number of elements in Q is: 0");
   return bFound;
}

/*------------------------------------------------------------------------------
| IDDEServerConversation::removeEvent                                          |
------------------------------------------------------------------------------*/
IDDEServerAcknowledgeEvent* IDDEServerConversation :: removeEvent ( )
{
   IMODTRACE_DEVELOP("IDDEServerConversation::removeEvent");
   if (pAckQCl)
   {
      const IDDEServerAcknowledgeEvent* pEvt;
      pEvt = pAckQCl->firstElement();
      pAckQCl->removeFirst();
      Boolean bEmpty = pAckQCl->isEmpty();
      if (bEmpty)
      {
         ITRACE_DEVELOP("Number of elements in Q is: 0");
         delete pAckQCl;
         pAckQCl = 0;
      }
      else
         ITRACE_DEVELOP("Number of elements in Q is: " +
            IString(pAckQCl->numberOfElements()));
      return (IDDEServerAcknowledgeEvent*)pEvt;
   }
   return 0;
}

/*------------------------------------------------------------------------------
| IDDEServerConversation::addEvent                                             |
------------------------------------------------------------------------------*/
void IDDEServerConversation :: addEvent ( IDDEServerAcknowledgeEvent* pEvt)
{
   IMODTRACE_DEVELOP("IDDEServerConversation::addEvent");
   if (pAckQCl == 0)
      pAckQCl = new IDDEAckQueue;
   pAckQCl->add(pEvt);
   ITRACE_DEVELOP("Number of elements in Q is: " +
      IString(pAckQCl->numberOfElements()));
}


/*------------------------------------------------------------------------------
| IDDEStatics::semaphor                                                        |
------------------------------------------------------------------------------*/
IPrivateResource& IDDEStatics :: semaphor ( )
{
   if (pPriResCl == 0)
   {
      pPriResCl = new IPrivateResource();
   }
   return *pPriResCl;
}

/*------------------------------------------------------------------------------
| IDDEStatics::~IDDEStatics                                                    |
------------------------------------------------------------------------------*/
IDDEStatics :: ~IDDEStatics ( )
{
   if (pPriResCl)
      delete pPriResCl;
}
