/*******************************************************************************
* FILE NAME: icnrctl9.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains static objects and misc functions for                   *
*   the class IContainerControl, declared in icnrctl.hpp.                      *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512 + IWindow(10) + IControl(10)
#pragma priority( -2147482092 )

#define INCL_WINSTDCNR
#define INCL_WINSTDDRAG

#include <icnrrec.hpp>  // Must be first for OS flags
#include <ithread.hpp>
#include <itrace.hpp>
#include <iexcept.hpp>
#include <ireslock.hpp>
#include <icnrctl.hpp>
#include <ievent.hpp>
#include <ihandler.hpp>
#include <icnrstat.hpp>


// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNRCTL9_CPP_
  #include <ipagetun.h>
#endif


// Forward declares
class IPrivateResource;
class ICnrDestroyHandler;
class IWindow;

//***************************************************************
// Public Container Styles.
//***************************************************************

#pragma data_seg(ICLStaticConst)
const IContainerControl::Style
  IContainerControl::readOnly          = CCS_READONLY,
  IContainerControl::autoPosition      = CCS_AUTOPOSITION,
  IContainerControl::extendedSelection = CCS_EXTENDSEL,
  IContainerControl::singleSelection   = CCS_SINGLESEL,
  IContainerControl::multipleSelection = CCS_MULTIPLESEL,
  IContainerControl::verifyPointers    = CCS_VERIFYPOINTERS,
  IContainerControl::noSharedObjects   (0, ICCS_NOSHAREDOBJECTS),
  IContainerControl::pmCompatible      (0, ICCS_PMCOMPATIBILITY),
  IContainerControl::classDefaultStyle ( WS_VISIBLE |
                                         CCS_SINGLESEL,
                                         0 );
#pragma data_seg()

#pragma data_seg(ICLNonConst)
  IContainerControl::Style
            IContainerControl::currentDefaultStyle ( WS_VISIBLE |
                                                     CCS_SINGLESEL,
                                                     0 );
#pragma data_seg()


//***************************************************************
// Public Container Attributes.
//***************************************************************
#pragma data_seg(ICLStaticConst)
const IContainerControl::Attribute
  IContainerControl::textView              = CV_TEXT,
  IContainerControl::iconView              = CV_ICON,
  IContainerControl::nameView              = CV_NAME,
  IContainerControl::detailsView           = CV_DETAIL,
  IContainerControl::treeView              = CV_TREE,
  IContainerControl::flowedView            = CV_FLOW,
  IContainerControl::miniIcons             = CV_MINI,
  IContainerControl::readOnlyTitle         = CA_TITLEREADONLY,
  IContainerControl::titleSeparator        = CA_TITLESEPARATOR,
  IContainerControl::detailsViewTitles     = CA_DETAILSVIEWTITLES,
  IContainerControl::visibleTitle          = CA_CONTAINERTITLE,
  IContainerControl::alignTitleCentered    = CA_TITLECENTER,
  IContainerControl::alignTitleLeft        = CA_TITLELEFT,
  IContainerControl::alignTitleRight       = CA_TITLERIGHT,
  IContainerControl::handleDrawItem        = CA_OWNERDRAW,
  IContainerControl::handleDrawBackground  = CA_OWNERPAINTBACKGROUND,
  IContainerControl::orderedTargetEmphasis = CA_ORDEREDTARGETEMPH,
  IContainerControl::mixedTargetEmphasis   = CA_MIXEDTARGETEMPH,
  IContainerControl::visibleTreeLine       = CA_TREELINE,
  IContainerControl::classDefaultAttribute = orderedTargetEmphasis |
                                             detailsViewTitles     |
                                             visibleTreeLine       |
                                             readOnlyTitle         |
                                             iconView;
#pragma data_seg()

#pragma data_seg(ICLNonConst)
  IContainerControl::Attribute
            IContainerControl::currentDefaultAttribute = classDefaultAttribute;

  // Static Pointer allocation
  static ICnrControlStaticPtr CnrCtlStatics;
#pragma data_seg()

IPrivateResource* ICnrControlStaticPtr::pCnrKey = 0;
ICnrDestroyHandler* ICnrControlStaticPtr::pHandler = 0;

ICnrDestroyHandler ::  ICnrDestroyHandler() { }
ICnrDestroyHandler :: ~ICnrDestroyHandler() { }


/*------------------------------------------------------------------------------
 Function Name: IContainerControl :: setAllocationContainer

 Implementation: set allocation container handle for this container.
 This is the "hidden" container in which all our other containers
  (for this thread) reside.
------------------------------------------------------------------------------*/
void IContainerControl :: setAllocationContainer(unsigned long hwndContainer)
{
   IMODTRACE_DEVELOP("CnrCtl::setAllocationContainer");
   IThread::current().setVariable("ICnrCtl::hwnd", IString(&hwndContainer,
                                                           sizeof(hwndContainer)));
   ITRACE_DEVELOP(IString("ICnrCTl::hwnd is: ") +
                  IThread::current().variable("ICnrCtl::hwnd"));
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl :: hwndAllocation

 Implementation: return allocation container handle for this container
------------------------------------------------------------------------------*/
IWindowHandle IContainerControl :: hwndAllocation()
{
   IString strAllocate = IThread::current().variable("ICnrCtl::hwnd");
   IWindowHandle whnd;

   if (strAllocate.length() != 0)
#ifdef IC_WIN
     whnd = (void*)*(unsigned long*)(char*)strAllocate;
#else
     whnd = *(unsigned long*)(char*)strAllocate;
#endif

#ifndef IC_MOTIF
   // If not allocated or an invalid handle, call initialize and try again
   if ((strAllocate.length()==0) ||
       (!IISWINDOW(IThread::current().anchorBlock(), whnd)))
#endif
#ifdef IC_MOTIF
   // If not allocated call initialize and try again
   if (strAllocate.length()==0)
#endif
   {
      IContainerControl::initialize();
      strAllocate = IThread::current().variable("ICnrCtl::hwnd");
      if (strAllocate.length() != 0)
#ifdef IC_WIN
        whnd = (void*)*(unsigned long*)(char*)strAllocate;
#else
        whnd = *(unsigned long*)(char*)strAllocate;
#endif
   }
   IASSERT(strAllocate.length()!=0);
   return whnd;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl :: containerKey

 Implementation: return the key (semaphore) used to serialize
 access to this container
------------------------------------------------------------------------------*/
IPrivateResource& IContainerControl ::containerKey()
{
   if(CnrCtlStatics.pCnrKey == 0)
   {
      CnrCtlStatics.pCnrKey = new IPrivateResource();
   }
   return *CnrCtlStatics.pCnrKey;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl :: destroyHandler

 Implementation:  return reference to the handler which
  processes WM_DESTROY for this container.
------------------------------------------------------------------------------*/
ICnrDestroyHandler& IContainerControl ::destroyHandler()
{
   if(CnrCtlStatics.pHandler == 0)
   {
      CnrCtlStatics.pHandler = new ICnrDestroyHandler();
   }
   return *CnrCtlStatics.pHandler;
}


unsigned long IContainerControl :: baseRecordSize()
{
   return sizeof(IMiniCnrRecord);
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: containerInfo

 Implementation: Private function to get and return the container's CNRINFO
                 structure as ICnrInfo.
------------------------------------------------------------------------------*/
void IContainerControl :: containerInfo(ICnrInfo* pCnrInfo) const
{
  Boolean fSuccess = (Boolean)(this->sendEvent(CM_QUERYCNRINFO,
                                               MPFROMP(pCnrInfo),
                                               MPFROMLONG(sizeof(ICnrInfo))));
  if (!fSuccess)
  {
    ITHROWGUIERROR("CM_QUERYCNRINFO");
  }
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setContainerInfo

 Implementation: Private function to set the container's CNRINFO structure.
------------------------------------------------------------------------------*/
void IContainerControl::setContainerInfo(ICnrInfo* pCnrInfo,
                                         unsigned long flags)
{
  Boolean fSuccess = (Boolean)(this->sendEvent(CM_SETCNRINFO,
                                               MPFROMP(pCnrInfo),
                                               MPFROMLONG(flags)));
  if (!fSuccess)
  {
    ITHROWGUIERROR("CM_SETCNRINFO");
  }
}

/***************************************************************/
/* Return the default style for new entry field objects.       */
/***************************************************************/
IContainerControl::Style  IContainerControl :: defaultStyle()
{
  return currentDefaultStyle;
}


/***************************************************************/
/* Replace the default style for new container controls.       */
/***************************************************************/
void  IContainerControl :: setDefaultStyle
                                  (const IContainerControl::Style& cnrStyle)
{
  currentDefaultStyle = cnrStyle;
}


/***************************************************************/
/* Returns base style for the control by default, or extended  */
/* style if extended flag (bExtOnly) is set.                   */
/***************************************************************/
unsigned long IContainerControl :: convertToGUIStyle(const IBitFlag& guiStyle,
                                                     Boolean bExtOnly) const
{
  // Obtain the style from the class (IControl) that we inherit from
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    ulStyle |= extendedStyle() & IS_EXTMASK;
  }
  else
  {
    // CCS_ styles have a one-to-one correspondence to our style bits, and
    // inhabit the lower word of the base GUI style.  Therefore, obtain
    // that portion asis, masking out the upper word.
    ulStyle |= guiStyle.asUnsignedLong() & ICCS_MASK;
#ifdef IC_WIN
    ulStyle |= CNR_REQUIREDSTYLES;
#endif
  }

  return( ulStyle );
}

/***************************************************************/
/* Returns true if we are using the CCL control on windows.    */
/* All other platforms return true.                            */
/***************************************************************/
Boolean IContainerControl::isPMCompatible( ) const
{
#ifdef IC_WIN
  return (Boolean)ppd->pmCompatible;
#else
  return true;
#endif
}

/***************************************************************/
/* Return the default attribute for new container controls.    */
/***************************************************************/
IContainerControl::Attribute  IContainerControl :: defaultAttribute()
{
  return currentDefaultAttribute;
}


/***************************************************************/
/* Replace the default style for new container controls.       */
/***************************************************************/
void  IContainerControl :: setDefaultAttribute
                            (const IContainerControl::Attribute& cnrAttribute)
{
  currentDefaultAttribute = cnrAttribute;
}


/*-----------------------------------------------------------------------------
 Function Name: ICnrDestroyHandler::dispatchHandlerEvent

 Implementation: handle WM_DESTROY
------------------------------------------------------------------------------*/
IBase::Boolean ICnrDestroyHandler::dispatchHandlerEvent(IEvent& evt)
{
   if (evt.eventId() == WM_DESTROY)
   {
//      IContainerControl* pcnrctl = (IContainerControl*)IWindow::windowWithHandle(evt.handle());
      IContainerControl* pcnrctl = (IContainerControl*)(evt.controlWindow());
      if(pcnrctl!=0)
      {
        pcnrctl->cleanUp();
      }
   }
   //---------------------------------------------------------------
   // DEFECT 23407 : ( for both IC_PM and IC_WIN )
   //                Force the mle to flush/close.
   //---------------------------------------------------------------
   if ( evt.eventId() == WM_HSCROLL || evt.eventId() == WM_VSCROLL )
   {
      // We need to make sure any posted CM_CLOSEEDIT is processed before
      // we continue with scrolling.  Remove any from the queue and SEND 
      // CM_CLOSEEDIT now.  We post this message in closeEdit().
      QMSG  qmsg;
      Boolean closeFound = false;
      IWindow* pwndControl = evt.controlWindow();
      while ( IPEEKMSG( IThread::current().anchorBlock(), &qmsg, 
                        pwndControl->handle(),
                        CM_CLOSEEDIT, CM_CLOSEEDIT, PM_REMOVE ) )
      {
        closeFound = true;
      }
      pwndControl->sendEvent(CM_CLOSEEDIT, 0, 0);
   }
   //---------------------------------------------------------------
   return false;
}


/*-----------------------------------------------------------------------------
 Function Name: ICnrDestroyHandler::handleEventsFor

 Implementation: enable this handler to handle events
 for the specified container.
------------------------------------------------------------------------------*/
ICnrDestroyHandler& ICnrDestroyHandler :: handleEventsFor( IContainerControl *container )
{
  IASSERTPARM(container != 0);
  IHandler::handleEventsFor(container);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: ICnrDestroyHandler::stopHandlingEventsFor

 Implementation: disable this handler from handling events for
 the specified container.
------------------------------------------------------------------------------*/
ICnrDestroyHandler& ICnrDestroyHandler :: stopHandlingEventsFor( IContainerControl *container )
{
  IASSERTPARM(container != 0);
  IHandler::stopHandlingEventsFor(container);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: ICnrDestroyHandler::handleEventsFor and
 ::stopHandlingEventsFor

 Implementation: insure that, if someone attempts to attach
 this handler to any other window or control type (besides
 an IContainerControl), the request is harmlessly ignored.
------------------------------------------------------------------------------*/
IHandler& ICnrDestroyHandler :: handleEventsFor       ( IWindow* window  )
  {return *this; }

IHandler& ICnrDestroyHandler :: stopHandlingEventsFor ( IWindow* window  )
  {return *this; }


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::addDefaultHandler

 Implementation: Add handler to container and increment frame count.
  If pHandler is null, create static handler.
------------------------------------------------------------------------------*/
void IContainerControl :: addDefaultHandler ( )
{
  destroyHandler().handleEventsFor(this);
  return;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::removeDefaultHandler

 Implementation: Remove default handler and decrement frame count.
  If container count goes to zero, delete static handler and zero pointer.
------------------------------------------------------------------------------*/
void IContainerControl :: removeDefaultHandler ( )
{
  destroyHandler().stopHandlingEventsFor(this);
  return;
}


IContainerControl& IContainerControl::setDeleteObjectsOnClose(IBase::Boolean fDestroy)
{
  if(fDestroy)
    flClState |= IContainerControl::autoDeleteObjects;
  else
    flClState &= ~IContainerControl::autoDeleteObjects;
  return *this;
}


IContainerControl& IContainerControl::setDeleteColumnsOnClose(IBase::Boolean fDestroy)
{
  if(fDestroy)
    flClState |= IContainerControl::autoDeleteColumns;
  else
    flClState &= ~IContainerControl::autoDeleteColumns;
  return *this;
}

