/*
 *  $Id: build.c,v 1.5 1997/05/19 12:29:58 mj Exp $
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *  Copyright (C) 1997 Martin Mares
 */

/*
 * This file builds a disk-image from three different files:
 *
 * - bootsect: exactly 512 bytes of 8086 machine code, loads the rest
 * - setup: 8086 machine code, sets up system parm
 * - system: 80386 code for actual system
 *
 * It does some checking that all files are of the correct type, and
 * just writes the result to stdout, removing headers and padding to
 * the right amount. It also writes some system data to stderr.
 */

/*
 * Changes by tytso to allow root device specification
 * High loaded stuff by Hans Lermen & Werner Almesberger, Feb. 1996
 * Cross compiling fixes by Gertjan van Wingerde, July 1996
 * Rewritten by Martin Mares, April 1997
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <unistd.h>
#include <fcntl.h>
#include <asm/boot.h>

typedef unsigned char byte;
typedef unsigned short word;
typedef unsigned long u32;

#define DEFAULT_MAJOR_ROOT 0
#define DEFAULT_MINOR_ROOT 0

/* Minimal number of setup sectors (see also bootsect.S) */
#define SETUP_SECTS 4

#define BOOT_SIZE 512

byte work_buf[1024];
byte boot_buf[BOOT_SIZE];

char* file_name[4];
int file_d[4];
u32 file_size[4];

int is_big_kernel;

void die(const char * str, ...)
{
	va_list args;
	va_start(args, str);
	vfprintf(stderr, str, args);
	fputc('\n', stderr);
	exit(1);
}

int file_open_rdonly(const char *name, u32* length)
{
	int fd;
	if ((fd = open(name, O_RDONLY, 0)) < 0)
		die("Unable to open `%s' for reading", name);
	*length = lseek(fd, 0, SEEK_END);
        lseek(fd, 0, SEEK_SET);
	return fd;
}

int file_open_write(const char* name)
{
	int fd;
	if ((fd = open(name, O_RDWR|O_CREAT, 0)) < 0)
		die("Unable to open `%s' for writing", name);
	return fd;
}

int file_close(int fd) 
{
	close(fd);
}

static int cp_buf_to_file(int fd_out, char* buf, int buf_size) 
{
	int w = write(fd_out, buf, buf_size);
	if (w < buf_size) 
		die("Unable to write to fd to %d\n", fd_out);
	return w;
}

static int cp_file_to_buf(char* buf, int buf_size, int fd_in, int block_size) 
{
	//assert block size <= buf size
	//assert bufsize divides block_size 
	int r = read(fd_in, buf, buf_size);
	if (r<buf_size){
		int b = (r+(block_size-1))/block_size;
	       	if (b*block_size > r) memset(buf+r, 0, b*block_size-r);
		return b*block_size;
	}
	return r;
}

static int file_write(int fd_out, char* buf, int buf_size, int fd_in, int file_size, int block_size) 
{
	int curr = file_size;
	while (curr > 0) {
		int s = cp_file_to_buf(buf, buf_size, fd_in, block_size);
		cp_buf_to_file(fd_out, buf, s);
		curr -= s;
	}
	
	return file_size - curr;
}


void usage(void)
{
	die("Usage: mkimage [-b] -[i|o] image bootsect setup system [rootdev]");
}

void output_mode(byte major_root, byte minor_root) {

	int i;
	int s;
	u32 setup_sectors;
	u32 sys_size;

	//initializeation
	file_d[0]= file_open_write(file_name[0]);
	for (i=1; i<4; i++) {
		file_d[i] = file_open_rdonly(file_name[i], &file_size[i]);
	}

	//boot sector read and sanity check
	s=cp_file_to_buf(boot_buf, sizeof(boot_buf), file_d[1], sizeof(boot_buf));
	if (boot_buf[510] != 0x55 || boot_buf[511] != 0xaa)
		die("Boot block hasn't got boot flag (0xAA55)");

	//setup sector sanity check 
	setup_sectors = (file_size[2]+(BOOT_SIZE-1))/BOOT_SIZE; /* Pad unused space with zeros */
	/* for compatibility with ancient versions of LILO. */
	if (setup_sectors < SETUP_SECTS)
		setup_sectors = SETUP_SECTS;
	fprintf(stderr, "Setup is %d bytes.\n", setup_sectors*BOOT_SIZE);

	//system image sanity check
	fprintf (stderr, "System is %d kB\n", file_size[3]/1024);
	sys_size = (file_size[3] + 15) / 16;
	/* 0x28000*16 = 2.5 MB, conservative estimate for the current maximum */
	if (sys_size > (is_big_kernel ? 0x28000 : DEF_SYSSIZE))
		die("System is too big. Try using %smodules.",
			is_big_kernel ? "" : "bzImage or ");
	if (sys_size > 0xefff)
		fprintf(stderr,"warning: kernel is too big for standalone boot "
		    "from floppy\n");

	//fix up the boot sector
	boot_buf[508] = minor_root;
	boot_buf[509] = major_root;
	boot_buf[497] = setup_sectors;
	boot_buf[500] = (sys_size & 0xff);
	boot_buf[501] = ((sys_size >> 8) & 0xff);

	//write everything out
	s=cp_buf_to_file(file_d[0], boot_buf, sizeof(boot_buf));
	fprintf(stderr, "boot   : %8d bytes\n", s);
	s=file_write(file_d[0], work_buf, sizeof(work_buf), file_d[2], file_size[2], BOOT_SIZE); 	
	fprintf(stderr, "setup  : %8d bytes\n", s);
	s=file_write(file_d[0], work_buf, sizeof(work_buf), file_d[3], file_size[3], 1); 	
	fprintf(stderr, "vmlinux: %8d bytes\n", s);
	
}

void input_mode(byte* major_root, byte* minor_root) {

	int i;
	int s;
	u32 setup_sectors;
	u32 sys_size1;
	u32 sys_size2;

	//initializeation
	file_d[0]= file_open_rdonly(file_name[0],&file_size[0]);
	for (i=1; i<4; i++) {
		file_d[i] = file_open_write(file_name[i]);
	}

	//boot sector read and sanity check
	s=file_write(file_d[1], boot_buf, sizeof(boot_buf), file_d[0], sizeof(boot_buf), sizeof(boot_buf));
	if (boot_buf[510] != 0x55 || boot_buf[511] != 0xaa)
		die("Boot block hasn't got boot flag (0xAA55)");
	//boot sector
	*minor_root = boot_buf[508];
	*major_root = boot_buf[509];
	setup_sectors = boot_buf[497];
	sys_size1 = 16*((boot_buf[500])|(boot_buf[501] << 8));
	sys_size2 = (file_size[0]-s)-(setup_sectors*BOOT_SIZE);

	//Sanity check
	if (abs((int)sys_size1-(int)sys_size2)>=16) 
		fprintf(stderr, "Size parameters do not match specification...corrupt image ?\n");


	//write everything out
	fprintf(stderr, "boot   : %8d bytes\n", s);
	s=file_write(file_d[2], work_buf, sizeof(work_buf), file_d[0], setup_sectors*BOOT_SIZE, BOOT_SIZE); 	
	fprintf(stderr, "setup  : %8d bytes\n", s);
	s=file_write(file_d[3], work_buf, sizeof(work_buf), file_d[0], sys_size2, 1); 	
	fprintf(stderr, "vmlinux: %8d bytes\n", s);
	
}

int main(int argc, char ** argv)
{
	int i;
	byte major_root, minor_root;
	struct stat sb;

	char mode = '\0';

	if (argc > 2 && !strcmp(argv[1], "-b"))
	{
	    is_big_kernel = 1;
	    argc--, argv++;
	}

	if ((argc < 6) || (argc > 7))
		usage();

	if (!strcmp(argv[1], "-i")) 
	{
		mode = 'i';	
	}
	if (!strcmp(argv[1], "-o")) 
	{
		mode = 'o';	
	}

	if (mode == '\0') 
		usage();

	if (argc > 6) {
		if (mode == 'i') 
			usage();

		if (!strcmp(argv[6], "CURRENT")) {
			if (stat("/", &sb)) {
				perror("/");
				die("Couldn't stat /");
			}
			major_root = major(sb.st_dev);
			minor_root = minor(sb.st_dev);
		} else if (strcmp(argv[6], "FLOPPY")) {
			if (stat(argv[6], &sb)) {
				perror(argv[6]);
				die("Couldn't stat root device.");
			}
			major_root = major(sb.st_rdev);
			minor_root = minor(sb.st_rdev);
		} else {
			major_root = 0;
			minor_root = 0;
		}
		fprintf(stderr, "Root device is (%d, %d)\n", major_root, minor_root);
	} else {
		major_root = DEFAULT_MAJOR_ROOT;
		minor_root = DEFAULT_MINOR_ROOT;
	}

	for (i=0; i<5; i++) {
		file_name[i] = argv[i+2];
	}

	switch (mode) {
		case 'i':
			input_mode(&major_root, &minor_root);
			fprintf(stderr, "Boot Device: (%2.2x,%2.2x)\n", major_root, minor_root);
			break;
			
		case 'o':
			output_mode(major_root, minor_root);
			break;
	}

	for (i=0; i<4; i++) {
		file_close(file_d[i]);
	}
}
